<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h2>PHP 8.2 Compatibility Checker for ERP System</h2>";
echo "PHP Version: " . PHP_VERSION . "<br><br>";

// Initialize counters
$issues_found = [];
$warnings = [];

// Check 1: Test DTO classes with 'var' keyword
echo "<h3>1. Checking DTO Classes (Dynamic Properties)</h3>";
try {
    require_once('../models/dto/User.class.php');
    $user = new User();
    $user->username = "test";
    echo "✓ DTO classes work (User class tested)<br>";
    echo "<small>Note: 'var' keyword is deprecated but still works. Consider migrating to 'public' later.</small><br>";
} catch (Error $e) {
    $issues_found[] = "DTO Classes: " . $e->getMessage();
    echo "✗ DTO issue: " . $e->getMessage() . "<br>";
}

// Check 2: Session handling
echo "<h3>2. Checking Session Handling</h3>";
try {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    $_SESSION['test'] = 'value';
    echo "✓ Session handling works<br>";
} catch (Error $e) {
    $issues_found[] = "Session: " . $e->getMessage();
    echo "✗ Session issue: " . $e->getMessage() . "<br>";
}

// Check 3: Database connection
echo "<h3>3. Checking Database Connection</h3>";
try {
    require_once('../models/sql/ConnectionProperty.class.php');
    $conn = @mysqli_connect(
        ConnectionProperty::getHost(),
        ConnectionProperty::getUser(),
        ConnectionProperty::getPassword(),
        ConnectionProperty::getDatabase()
    );
    if ($conn) {
        echo "✓ Database connection works<br>";
        mysqli_close($conn);
    } else {
        $warnings[] = "Database connection failed - check credentials";
        echo "⚠ Database connection failed<br>";
    }
} catch (Error $e) {
    $issues_found[] = "Database: " . $e->getMessage();
    echo "✗ Database issue: " . $e->getMessage() . "<br>";
}

// Check 4: RedBeanPHP
echo "<h3>4. Checking RedBeanPHP ORM</h3>";
try {
    require_once('../library/db_main/rb.php');
    echo "✓ RedBeanPHP loads successfully<br>";
} catch (Error $e) {
    $issues_found[] = "RedBeanPHP: " . $e->getMessage();
    echo "✗ RedBeanPHP issue: " . $e->getMessage() . "<br>";
}

// Check 5: Smarty
echo "<h3>5. Checking Smarty Template Engine</h3>";
try {
    require_once('../public/libs/Smarty.class.php');
    $smarty = new Smarty();
    echo "✓ Smarty template engine works<br>";
} catch (Error $e) {
    $issues_found[] = "Smarty: " . $e->getMessage();
    echo "✗ Smarty issue: " . $e->getMessage() . "<br>";
}

// Check 6: Scan for deprecated functions
echo "<h3>6. Scanning for Deprecated Functions</h3>";
$deprecated_functions = ['each', 'create_function', 'ereg', 'split', 'mysql_connect', 'mysql_query'];
$files_to_check = [
    '../controllers/index.php',
    '../controllers/login.php',
    '../public/authentication.php'
];

$found_deprecated = false;
foreach ($files_to_check as $file) {
    if (file_exists($file)) {
        $content = file_get_contents($file);
        foreach ($deprecated_functions as $func) {
            if (strpos($content, $func . '(') !== false) {
                $warnings[] = "Found deprecated function '$func' in " . basename($file);
                $found_deprecated = true;
            }
        }
    }
}

if (!$found_deprecated) {
    echo "✓ No deprecated functions found in main controllers<br>";
} else {
    echo "⚠ Some deprecated functions found (see warnings below)<br>";
}

// Summary
echo "<hr><h3>Summary</h3>";

if (empty($issues_found)) {
    echo "<p style='color: green; font-size: 18px; font-weight: bold;'>✓ No critical issues found! Your ERP is PHP 8.2 compatible!</p>";
} else {
    echo "<p style='color: red; font-size: 18px; font-weight: bold;'>✗ Critical Issues Found:</p>";
    echo "<ul style='color: red;'>";
    foreach ($issues_found as $issue) {
        echo "<li>$issue</li>";
    }
    echo "</ul>";
}

if (count($warnings) > 0) {
    echo "<p style='color: orange;'><strong>Warnings:</strong></p>";
    echo "<ul style='color: orange;'>";
    foreach ($warnings as $warning) {
        echo "<li>$warning</li>";
    }
    echo "</ul>";
}

echo "<hr>";
echo "<p><strong>Next Steps:</strong></p>";
echo "<ul>";
echo "<li><a href='test_complete.php'>Run Complete System Test</a></li>";
echo "<li><a href='login.php'>Go to Login Page</a></li>";
echo "<li><a href='test_index.php'>Test Index Page</a></li>";
echo "</ul>";
?>
