<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h2>Scanning for count() Issues (PHP 8.2)</h2>";
echo "<p>Looking for count() called on objects that might not be arrays...</p><hr>";

$controllers_dir = '/Applications/AMPPS/www/erp19/controllers/';
$potential_issues = [];

// Get all PHP files
$files = glob($controllers_dir . '*.php');

foreach ($files as $file) {
    $basename = basename($file);
    
    // Skip test/scan files
    if (strpos($basename, 'test_') === 0 || strpos($basename, 'scan_') === 0 || 
        strpos($basename, 'check_') === 0 || $basename === 'FINAL_TEST.php') {
        continue;
    }
    
    $content = file_get_contents($file);
    $lines = explode("\n", $content);
    
    foreach ($lines as $line_num => $line) {
        $actual_line_num = $line_num + 1;
        
        // Look for patterns like: count($someData) where $someData might be a single object
        // Specifically looking for: count($xxxData) or count($xxxRecord) patterns
        if (preg_match('/count\(\$(\w+(?:Data|Record|Setting|Config|Info))\)/', $line, $matches)) {
            $var_name = $matches[1];
            
            // Check if this variable is loaded from DAO (likely single object)
            $before_content = implode("\n", array_slice($lines, max(0, $line_num - 20), 20));
            
            // Look for patterns indicating single object load
            $is_dao_load = preg_match('/\$' . preg_quote($var_name) . '\s*=\s*\$\w+DAO->load\(/', $before_content);
            $is_single_result = preg_match('/\$' . preg_quote($var_name) . '\s*=\s*\w+\(\)/', $before_content);
            
            if ($is_dao_load || $is_single_result) {
                $potential_issues[] = [
                    'file' => $basename,
                    'line' => $actual_line_num,
                    'variable' => $var_name,
                    'code' => trim($line),
                    'likely_object' => $is_dao_load ? 'DAO->load()' : 'function return'
                ];
            }
        }
    }
}

// Display results
if (count($potential_issues) > 0) {
    echo "<h3 style='color: orange;'>⚠ Found " . count($potential_issues) . " potential count() issues:</h3>";
    echo "<p>These may cause <strong>TypeError</strong> in PHP 8.2 if the variable is an object instead of an array.</p>";
    
    echo "<table border='1' cellpadding='10' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr style='background: #f0f0f0;'><th>File</th><th>Line</th><th>Variable</th><th>Likely Type</th><th>Code</th></tr>";
    
    foreach ($potential_issues as $issue) {
        echo "<tr>";
        echo "<td>{$issue['file']}</td>";
        echo "<td>{$issue['line']}</td>";
        echo "<td style='color: red;'>\${$issue['variable']}</td>";
        echo "<td style='color: orange;'>{$issue['likely_object']}</td>";
        echo "<td><code>" . htmlspecialchars($issue['code']) . "</code></td>";
        echo "</tr>";
    }
    
    echo "</table>";
    
    echo "<hr>";
    echo "<h3>Fix Recommendation:</h3>";
    echo "<p>Replace count() checks on single objects with:</p>";
    echo "<pre style='background: #f0f0f0; padding: 10px;'>";
    echo "// OLD (causes TypeError in PHP 8.2)\n";
    echo "if (count(\$objectData) > 0) { ... }\n\n";
    echo "// NEW (PHP 8.2 compatible)\n";
    echo "if (!empty(\$objectData) && isset(\$objectData->someProperty)) { ... }\n";
    echo "// OR simply\n";
    echo "if (!empty(\$objectData)) { ... }\n";
    echo "</pre>";
    
} else {
    echo "<h3 style='color: green;'>✓ No obvious count() issues found!</h3>";
}

echo "<hr>";
echo "<p><a href='productCatController.php?do=edit&id=2&parentid=0'>Test Product Category Edit</a> | ";
echo "<a href='FINAL_TEST.php'>Run Final Test</a></p>";
?>
