<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h2>Scanning for PHP 8.2 Null Object Assignment Issues</h2>";
echo "<p>Looking for variables used without initialization...</p><hr>";

$controllers_dir = '/Applications/AMPPS/www/erp19/controllers/';
$issues_found = [];

// Get all PHP files in controllers
$files = glob($controllers_dir . '*.php');

foreach ($files as $file) {
    $basename = basename($file);
    
    // Skip test files
    if (strpos($basename, 'test_') === 0 || strpos($basename, 'scan_') === 0 || strpos($basename, 'check_') === 0 || $basename === 'FINAL_TEST.php') {
        continue;
    }
    
    $content = file_get_contents($file);
    $lines = explode("\n", $content);
    
    foreach ($lines as $line_num => $line) {
        $actual_line_num = $line_num + 1;
        
        // Look for patterns like: $obj->property = value (without prior $obj = new ...)
        // This is a simplified check
        if (preg_match('/^\s*\$(\w+)->\w+\s*=/', $line, $matches)) {
            $var_name = $matches[1];
            
            // Skip common globals and known safe variables
            if (in_array($var_name, ['smarty', '_GET', '_POST', '_SESSION', '_SERVER', 'this'])) {
                continue;
            }
            
            // Check if this variable was initialized earlier in the file
            $before_content = implode("\n", array_slice($lines, 0, $line_num));
            
            // Look for initialization patterns
            $has_new = preg_match('/\$' . preg_quote($var_name) . '\s*=\s*new\s+/', $before_content);
            $has_load = preg_match('/\$' . preg_quote($var_name) . '\s*=\s*\$\w+->load\(/', $before_content);
            $has_array_access = preg_match('/\$' . preg_quote($var_name) . '\s*=\s*\$\w+\[\d+\]/', $before_content);
            $has_stdclass = preg_match('/\$' . preg_quote($var_name) . '\s*=\s*new\s+stdClass/', $before_content);
            
            if (!$has_new && !$has_load && !$has_array_access && !$has_stdclass) {
                $issues_found[] = [
                    'file' => $basename,
                    'line' => $actual_line_num,
                    'variable' => $var_name,
                    'code' => trim($line)
                ];
            }
        }
    }
}

// Display results
if (count($issues_found) > 0) {
    echo "<h3 style='color: orange;'>⚠ Found " . count($issues_found) . " potential issues:</h3>";
    echo "<table border='1' cellpadding='10' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr style='background: #f0f0f0;'><th>File</th><th>Line</th><th>Variable</th><th>Code</th></tr>";
    
    foreach ($issues_found as $issue) {
        echo "<tr>";
        echo "<td>{$issue['file']}</td>";
        echo "<td>{$issue['line']}</td>";
        echo "<td style='color: red;'>\${$issue['variable']}</td>";
        echo "<td><code>" . htmlspecialchars($issue['code']) . "</code></td>";
        echo "</tr>";
    }
    
    echo "</table>";
    
    echo "<hr>";
    echo "<h3>Recommendation:</h3>";
    echo "<p>Each of these variables should be initialized before use. Add one of:</p>";
    echo "<ul>";
    echo "<li><code>\$variable = new ClassName();</code> (for DTO objects)</li>";
    echo "<li><code>\$variable = new stdClass();</code> (for generic objects)</li>";
    echo "<li>Check if variable is loaded from database first</li>";
    echo "</ul>";
    
} else {
    echo "<h3 style='color: green;'>✓ No obvious issues found!</h3>";
    echo "<p>All variables appear to be initialized before property assignment.</p>";
}

echo "<hr>";
echo "<p><a href='FINAL_TEST.php'>Run Final Test</a> | <a href='login.php'>Go to Login</a></p>";
?>
