<?php
// Debug version that captures all output and errors
header('Content-Type: application/json');

// Capture all errors
$errors = [];
$warnings = [];
$output = [];

set_error_handler(function($errno, $errstr, $errfile, $errline) use (&$errors, &$warnings) {
    $error = [
        'type' => $errno,
        'message' => $errstr,
        'file' => basename($errfile),
        'line' => $errline
    ];
    
    if (in_array($errno, [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR])) {
        $errors[] = $error;
    } else {
        $warnings[] = $error;
    }
    return true;
});

register_shutdown_function(function() use (&$errors) {
    $error = error_get_last();
    if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        $errors[] = [
            'type' => 'FATAL',
            'message' => $error['message'],
            'file' => basename($error['file']),
            'line' => $error['line']
        ];
    }
});

// Start output buffering to capture the actual response
ob_start();

try {
    // Include the actual controller
    include('sellbillAjaxController.php');
    $actualOutput = ob_get_clean();
    
    // Try to decode if it's JSON
    $jsonData = json_decode($actualOutput, true);
    
    // Send debug response
    echo json_encode([
        'success' => true,
        'errors' => $errors,
        'warnings' => $warnings,
        'output_length' => strlen($actualOutput),
        'output_first_100' => substr($actualOutput, 0, 100),
        'is_valid_json' => (json_last_error() === JSON_ERROR_NONE),
        'json_error' => json_last_error_msg(),
        'actual_response' => $actualOutput,
        'decoded_data' => $jsonData,
        'get_params' => $_GET,
        'post_params' => $_POST,
        'request_params' => $_REQUEST
    ], JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    $actualOutput = ob_get_clean();
    
    echo json_encode([
        'success' => false,
        'exception' => [
            'message' => $e->getMessage(),
            'file' => basename($e->getFile()),
            'line' => $e->getLine(),
            'trace' => array_slice($e->getTrace(), 0, 5)
        ],
        'errors' => $errors,
        'warnings' => $warnings,
        'output_captured' => $actualOutput
    ], JSON_PRETTY_PRINT);
}
?>
