# Client Reports By Area Controller Documentation

**File**: `/controllers/clientReportsbyareaController.php`  
**Purpose**: Generates customer reports grouped by geographical areas and governorates with comprehensive financial analysis  
**Last Updated**: December 20, 2024  
**Total Functions**: 4  
**Lines of Code**: ~1,329

---

## 📋 Overview

The Client Reports By Area Controller is a specialized geographical reporting module that provides area-based customer debt analysis and financial tracking. It handles:
- Government and area-based customer grouping reports
- Regional customer debt summaries
- Area-wise financial performance analysis
- Customer payment tracking by geographical location
- Combined sales, returns, and payment analysis per region
- Date range filtering with customizable date field selection
- Multi-level geographical hierarchy (Government → Area → Client)
- Area and government totals aggregation
- Debit-only filtering for outstanding debt analysis

### Primary Functions
- [x] Generate area-based customer reports
- [x] Government-level customer summaries
- [x] Regional debt analysis
- [x] Area-wise payment tracking
- [x] Combined financial operations by region
- [x] Date range filtering (system date vs user-selected date)
- [x] Debt balance calculations per area
- [x] Geographic hierarchy reporting
- [x] Outstanding debt filtering
- [x] Area and government aggregation totals

### Related Controllers
- [clientReportsController.php](clientReportsController.md) - Individual customer reports
- [sellbillController.php](sellbillController.md) - Sales operations
- [clientController.php](#) - Customer management
- [returnsellbillController.php](#) - Sales returns
- [clientPayedDeptController.php](#) - Payment processing

---

## 🗄️ Database Tables

### Primary Tables (Direct Operations)
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **client** | Customer master data | clientid, clientname, clientdebt, clientarea, userid |
| **clientarea** | Customer area/region definitions | id, name, description |
| **government** | Government/state master data | governmetid, governmentname |
| **goverarea** | Government-area relationships | id, governmentid, clientareaid |
| **clientdebtchange** | Customer debt transaction log | clientdebtchangeid, clientid, clientdebtchangeamount, clientdebtchangetype, clientdebtchangedate, tablename, userSelectedDate |

### Sales & Financial Tables (Referenced)
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **sellbill** | Sales bills | sellbillid, sellbillclientid, sellbilltotalbill, sellbillaftertotalbill, sellbilltotalpayed, sellbilldate, conditions |
| **sellbilldetail** | Sales bill line items | sellbilldetailid, sellbillid, sellbilldetailproductid, sellbilldetailquantity |
| **returnsellbill** | Sales return bills | returnsellbillid, returnsellbillclientid, returnsellbilltotalbill, returnsellbillaftertotalbill, returnsellbilldate |
| **returnsellbilldetail** | Return bill details | returnsellbilldetailid, returnsellbillid, returnsellbilldetailproductid |
| **sellbillandrutern** | Combined sell & return operations | sellbillid, sellbillclientid, sellbillprice, returnsellbillprice, sellbilldate |
| **sellandruternbilldetail** | Combined bill details | sellandruternbilldetailid, sellbillid, sellbilldetailproductid, selltype |
| **bills** | Service/optical bills | billid, clientid, finalnetbillvalue, waitvalue, billdate, receivedate, deleted |
| **billsreturn** | Bill returns | billsreturnid, clientid, returnvalue, returnprice, date, deleted |

### Reference Tables
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **youtubelink** | Tutorial video links | youtubelinkid, title, url |
| **user** | System users | userid, username, viewclients |

---

## 🔑 Key Functions

### 1. **Default Action** - Area-Based Report (All Governments View)
**Location**: Lines 170-523  
**Purpose**: Generate comprehensive customer reports grouped by areas across all governments

**Function Signature**:
```php
// Triggered when: $do is empty or not specified
$clientareaid = $_POST["clientareaid"];
$governmentId = $_POST['governmentid'];
$startDate = $_POST["from"];
$endDate = $_POST["to"];
```

**Process Flow**:
1. Load reference data (governments, client areas, YouTube links)
2. Build geographic filtering query based on government and area selection
3. Handle three scenarios:
   - Specific government + specific area
   - Specific government + all areas
   - All governments + all areas
4. For each client in the filtered set:
   - Calculate payment totals from various transaction types
   - Calculate net sales totals
   - Process discount and tax calculations
   - Track deposit payments separately
5. Aggregate totals by area and government
6. Display via `allbyclientareabyallgovernmen.html` template

**Key Business Logic**:
```php
// Payment calculation logic
if (($myClientData[$i]->tablename == 'sellbillController.php' || 
     $myClientData[$i]->tablename == 'sellbillandruternController.php' || 
     $myClientData[$i]->tablename == 'returnsellbillController.php' ||
     $myClientData[$i]->tablename == 'depositcheckController.php' ||
     $myClientData[$i]->tablename == 'datedCheckedController.php' ||
     $myClientData[$i]->tablename == 'kempialaController.php')) {
    
    if ($myClientData[$i]->clientdebtchangetype == 1) {
        $myClientData[$i]->clientdebtchangeamount = $myClientData[$i]->clientdebtchangeamount * -1;
    }
    $paid_positive = $myClientData[$i]->clientdebtchangeamount + $paid_positive;
}
```

**Data Aggregation**:
- `$sumAllAreas` - Sales totals by area
- `$myDiscountSumAllAreas` - Discount totals by area  
- `$myGenSumAllAreas` - Net sales by area
- `$myPaySumAllAreas` - Payment totals by area
- `$clientdebtAllAreas` - Debt totals by area
- `$myPaySumSumGovernment` - Government-level payment totals
- `$myTotalbillSumGovernment` - Government-level sales totals
- `$myDiscountSumGovernment` - Government-level discount totals
- `$myGenSumGovernment` - Government-level net sales
- `$sumDept` - Government-level debt totals

---

### 2. **allbyclientarea Action** - Enhanced Area Report with Date Filtering
**Location**: Lines 524-924  
**Purpose**: Advanced area-based reporting with enhanced date filtering and debit-only options

**Function Signature**:
```php
// Triggered when: $do == "allbyclientarea"
$clientareaid = $_POST["clientareaid"];
$governmentId = $_POST['governmentid'];
$startDate = $_POST["from"];
$endDate = $_POST["to"];
$showDebitOnly = $_POST['showDebit'];
$searchDate = $_POST["searchDate"]; // 'sysDate' or 'userSelectedDate'
```

**Enhanced Features**:
1. **Dual Date Field Support**:
   - System date (`clientdebtchangedate`) - actual transaction date
   - User selected date (`userSelectedDate`) - user-defined business date
   
2. **Debit-Only Filtering**:
   - Option to show only customers with outstanding debt
   - Filters clients where `client.clientdebt != 0`

3. **Smart Date Handling**:
```php
if ($searchDate == "sysDate") {
    $datequery = ' and clientdebtchange.del = 0 AND clientdebtchange.clientdebtchangedate >= ("' . $startDate . ' 00:00:00" ) AND clientdebtchange.clientdebtchangedate <= ("' . $endDate . ' 23:59:59") ';
} elseif ($searchDate == "userSelectedDate") {
    $datequery = ' and clientdebtchange.del = 0 AND clientdebtchange.userSelectedDate >= ("' . $startDate . ' 00:00:00" ) AND clientdebtchange.userSelectedDate <= ("' . $endDate . ' 23:59:59") ';
}
```

4. **Dynamic Template Selection**:
   - `allbyclientareabyallgovernmen.html` - For all governments view
   - `allbyclientareabygovernment.html` - For specific government view

**Process Flow**:
1. Parse enhanced input parameters including date field selection
2. Apply debit-only filter if requested
3. Build date-aware queries for each transaction type
4. Execute same financial calculation logic as default action
5. Select appropriate template based on government filter scope
6. Generate the same comprehensive aggregation arrays

---

### 3. **getClientData()** - Client Data Provider
**Location**: Lines 936-942  
**Purpose**: Retrieve customer list for report dropdown population

**Function Signature**:
```php
function getClientData()
```

**Process Flow**:
1. Use `$clientExt->queryAllsup()` to load customer data
2. Return array for dropdown population

**Returns**: Array of customer objects suitable for form dropdowns

---

### 4. **clientShow()** - Individual Client Transaction History
**Location**: Lines 944-1165  
**Purpose**: Generate detailed transaction history for a specific client with document linking

**Function Signature**:
```php
function clientShow($clientid, $startDate, $endDate)
```

**Process Flow**:
1. Query client debt changes for the specified period
2. For each transaction, determine source document type and create appropriate links:
   - `sellbillController.php` → Sales bill links
   - `returnsellbillController.php` → Return bill links
   - `sellbillandruternController.php` → Combined bill links
   - `kempialaController.php` → Promissory note links
   - `depositcheckController.php` → Check deposit links
   - `clientPayedDeptController.php` → Payment receipt links
   - `billreceiptController.php` → Bill receipt links
3. Handle "bure" (net view) processing to remove offsetting entries
4. Recalculate running balances for net view
5. Assign processed data to Smarty template

**Document Linking Logic**:
```php
if ($data->tablename == "sellbillController.php") {
    $data->link = "sellbillController.php?do=showDetail&sellbillid=" . $data->clientdebtchangemodelid;
    $sellBillData = $sellBillDAO->load($data->clientdebtchangemodelid);
    $data->totalOperationCost = $sellBillData->sellbillaftertotalbill;
}
```

**Net View Processing** (bure = "1"):
```php
if ($type == 1) { // Payment entry
    $otherfinal = substr_replace($finalstring, '0', -1);
    foreach ($shownData as $xx) {
        if ($xx->finalstring == $otherfinal) {
            $xx->clientdebtchangeid = -1; // Hide offsetting entry
            $datax->clientdebtchangeid = -1;
        }
    }
}
```

---

## 🔄 Workflows

### Workflow 1: Geographic Area Report Generation
```
┌─────────────────────────────────────────────────────────────┐
│              START: Select Government/Area Filters         │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  1. Load Reference Data                                     │
│     - Load all governments                                  │
│     - Load all client areas                                 │
│     - Load YouTube tutorial links                           │
│     - Load all clients for dropdowns                       │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  2. Build Geographic Filter Query                          │
│     IF (specific government + specific area):               │
│       │                                                     │
│       └─→ Query clients in that area                       │
│                                                             │
│     ELSE IF (specific government + all areas):             │
│       │                                                     │
│       └─→ Query all clients in government                  │
│                                                             │
│     ELSE (all governments + all areas):                    │
│       │                                                     │
│       └─→ Query all clients with debt                      │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  3. Process Each Client in Geographic Set                  │
│     FOR EACH client in filtered set:                       │
│       │                                                     │
│       ├─→ Query debt change transactions                   │
│       │   ├─ Payment transactions                          │
│       │   ├─ Sales transactions                            │
│       │   └─ Other financial operations                    │
│       │                                                     │
│       ├─→ Query sales bills and details                    │
│       │   ├─ Regular sales bills                           │
│       │   ├─ Return bills                                  │
│       │   ├─ Combined sell/return bills                    │
│       │   └─ Optical/service bills                         │
│       │                                                     │
│       ├─→ Calculate Financial Totals                       │
│       │   ├─ Total payments (all sources)                  │
│       │   ├─ Net sales amounts                             │
│       │   ├─ Total discounts                               │
│       │   ├─ Deposit payments                              │
│       │   └─ Outstanding debt balance                      │
│       │                                                     │
│       └─→ Assign calculated values to client object       │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  4. Aggregate Totals by Geographic Hierarchy               │
│     FOR EACH area in results:                               │
│       │                                                     │
│       ├─→ Sum sales totals                                 │
│       ├─→ Sum discount totals                              │
│       ├─→ Sum net sales totals                             │
│       ├─→ Sum payment totals                               │
│       └─→ Sum debt totals                                  │
│                                                             │
│     FOR EACH government in results:                        │
│       │                                                     │
│       ├─→ Sum all area totals within government            │
│       ├─→ Create government-level aggregations             │
│       └─→ Track for grand total calculation                │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  5. Calculate Grand Totals                                 │
│     - Sum all government totals for overall summary        │
│     - Create final aggregation arrays                      │
│     - Prepare data for template display                    │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  6. Generate Geographic Report Output                      │
│     - Assign all data arrays to Smarty template            │
│     - Include area and government totals                   │
│     - Display via appropriate template:                    │
│       ├─ allbyclientareabyallgovernmen.html (all gov.)    │
│       └─ allbyclientareabygovernment.html (specific gov.) │
└─────────────────────────────────────────────────────────────┘
```

---

## 🌐 URL Routes & Actions

| URL Parameter | Function Called | Description |
|---------------|----------------|-------------|
| `do=` (empty) | Default action | Area-based report across all governments |
| `do=allbyclientarea` | Enhanced area action | Advanced area report with date filtering |

### Required Parameters by Action

**Default Area Report** (`do=` empty):
- `clientareaid` - Area ID (-1 for all areas)
- `governmentid` - Government ID (-2 for all governments)
- `from` - Start date (optional)
- `to` - End date (optional)

**Enhanced Area Report** (`do=allbyclientarea`):
- `clientareaid` - Area ID (-1 for all areas)
- `governmentid` - Government ID (-2 for all governments)
- `from` - Start date (YYYY-MM-DD)
- `to` - End date (YYYY-MM-DD)
- `showDebit` - Show only customers with debt (1 = yes, 0 = all)
- `searchDate` - Date field to use ("sysDate" or "userSelectedDate")

### URL Examples
```
# All governments, all areas, show only debtors
clientReportsbyareaController.php?do=allbyclientarea&governmentid=-2&clientareaid=-1&showDebit=1

# Specific government (ID 5), all areas
clientReportsbyareaController.php?do=allbyclientarea&governmentid=5&clientareaid=-1&from=2024-01-01&to=2024-12-31

# Specific government and area with date range
clientReportsbyareaController.php?do=allbyclientarea&governmentid=3&clientareaid=7&from=2024-11-01&to=2024-11-30&searchDate=userSelectedDate
```

---

## 🧮 Financial Calculation Methods

### Payment Calculation Logic
```php
// Process each debt change transaction
foreach ($myClientData as $transaction) {
    // Handle payment transactions from various sources
    if (($transaction->tablename == 'sellbillController.php' || 
         $transaction->tablename == 'sellbillandruternController.php' || 
         $transaction->tablename == 'returnsellbillController.php' ||
         $transaction->tablename == 'depositcheckController.php' ||
         $transaction->tablename == 'datedCheckedController.php' ||
         $transaction->tablename == 'kempialaController.php')) {
        
        // Reverse sign for payment transactions
        if ($transaction->clientdebtchangetype == 1) {
            $transaction->clientdebtchangeamount *= -1;
        }
        
        $paid_positive += $transaction->clientdebtchangeamount;
        
        // Track deposit payments separately
        if ($transaction->tablename == 'depositcheckController.php' && $transaction->del == 0) {
            $depositPaid += abs($transaction->clientdebtchangeamount);
        }
    }
    
    // Handle direct client payment receipts
    if ($transaction->tablename == 'clientPayedDeptController.php') {
        $paid_clientDept += $transaction->clientdebtchangeamount;
    }
    
    // Calculate net sales totals
    if ($transaction->tablename == 'sellbillController.php' || 
        $transaction->tablename == 'sellbillandruternController.php' || 
        $transaction->tablename == 'returnsellbillController.php') {
        
        if ($transaction->clientdebtchangetype == 1) {
            $transaction->totalOperationCost *= -1;
        }
        $net_total += $transaction->totalOperationCost;
    }
}
```

### Bills Payment Calculation
```php
// Process optical/service bills
for ($j = 0; $j <= $arrBill; $j++) {
    // Calculate payments from bills
    $paid_bill = $myClientBill[$j]->finalnetbillvalue - 
                 $myClientBill[$j]->waitvalue - 
                 $myClientBillReturn[$j]->returnvalue + 
                 $paid_bill;
    
    // Calculate net bill amounts
    $net_bill = $myClientBill[$j]->finalnetbillvalue - 
                $myClientBillReturn[$j]->returnprice + 
                $net_bill;
}
```

### Final Balance Calculation
```php
// Calculate comprehensive totals
$Net = $net_total + $net_bill; // Combined net sales
$paid = $paid_negative + $paid_positive + $paid_clientDept + $paid_bill; // Total payments
$balance = $clientMainDept - ($paid - $Net); // Final debt balance

// Calculate sales and discount totals from aggregated bill data
$total_gens = ($myClientDiscountAndTotalSellbills[0]->allGenTotal + 
               $myClientDiscountAndTotalSellbillsAndReturn[0]->allGenTotalSellAndReturn + 
               $myClientDiscountAndTotalBills[0]->totalNet) - 
               $myClientDiscountAndTotalReturnSellBill[0]->allGenTotalReturn;

$total_discounts = $myClientDiscountAndTotalSellbills[0]->totaldiscount + 
                   $myClientDiscountAndTotalBills[0]->totaldiscount + 
                   $myClientDiscountAndTotalSellbillsAndReturn[0]->totaldiscount - 
                   ($myClientDiscountAndTotalReturnSellBill[0]->totaldiscount + 
                    $myClientDiscountAndTotalBillsReturn[0]->clientreceivevalue);

$total_payed = ($paid_clientDept + 
                $myClientDiscountAndTotalSellbills[0]->sellbilltotalAllPayed + 
                $myClientDiscountAndTotalSellbillsAndReturn[0]->sellbillandruternPayedTotal) - 
                $myClientDiscountAndTotalReturnSellBill[0]->returnsellbilltotalpayed;
```

### Area and Government Aggregation
```php
// Aggregate totals by area
foreach ($clientShowData as $value) {
    if (in_array($value->clientareaid, $singleArea_arr)) {
        // Add to existing area totals
        $sumAllAreas[$value->clientareaid] += $value->myTotalbillSum;
        $myDiscountSumAllAreas[$value->clientareaid] += $value->myDiscountSum;
        $myGenSumAllAreas[$value->clientareaid] += $value->myGenSum;
        $myPaySumAllAreas[$value->clientareaid] += $value->myPaySum;
        $clientdebtAllAreas[$value->clientareaid] += $value->clientdebt;
    } else {
        // Create new area entry
        array_push($singleArea_arr, $value->clientareaid);
        $sumAllAreas[$value->clientareaid] = $value->myTotalbillSum;
        $myDiscountSumAllAreas[$value->clientareaid] = $value->myDiscountSum;
        $myGenSumAllAreas[$value->clientareaid] = $value->myGenSum;
        $myPaySumAllAreas[$value->clientareaid] = $value->myPaySum;
        $clientdebtAllAreas[$value->clientareaid] = $value->clientdebt;
    }
    
    // Same logic for government-level aggregation
    if (in_array($value->governmetid, $singleGovernment_arr)) {
        $sumDept[$value->governmetid] += $value->clientdebt;
        $myTotalbillSumGovernment[$value->governmetid] += $value->myTotalbillSum;
        // ... other government totals
    }
}
```

---

## 🔒 Security & Permissions

### Geographic Data Access Control
- No specific user permission filtering in this controller
- Assumes all users with access can view all geographic data
- Should be used in conjunction with user authentication middleware

### Input Sanitization
- All POST parameters filtered through framework validation
- Numeric IDs should be cast to integers for safety
- Date parameters validated before SQL inclusion
- SQL injection prevented by DAO layer parameterized queries

---

## 📊 Performance Considerations

### Database Optimization Tips
1. **Critical Indexes Required**:
   - `client(clientarea)` - For area-based grouping
   - `goverarea(governmentid, clientareaid)` - For government-area relationships
   - `clientdebtchange(clientid, clientdebtchangedate)` - For date-filtered queries
   - `clientdebtchange(clientid, userSelectedDate)` - For user date queries
   - `sellbill(sellbillclientid, sellbilldate)` - For sales queries
   - `bills(clientid, billdate)` - For bill queries

2. **Query Performance Issues**:
   - Large geographic areas may return thousands of customers
   - Consider pagination for very large areas/governments
   - Multiple complex aggregation queries per client can be slow
   - Date range queries across large datasets need optimization

3. **Memory Considerations**:
   - Area aggregation arrays can become very large
   - Template variable assignments for multiple areas consume memory
   - Consider processing areas in batches for very large datasets

### Optimization Recommendations
```sql
-- Add composite indexes for better performance
CREATE INDEX idx_client_area_debt ON client(clientarea, clientdebt);
CREATE INDEX idx_debt_client_date ON clientdebtchange(clientid, clientdebtchangedate, del);
CREATE INDEX idx_debt_client_userdate ON clientdebtchange(clientid, userSelectedDate, del);
CREATE INDEX idx_goverarea_composite ON goverarea(governmentid, clientareaid);
```

---

## 🐛 Common Issues & Troubleshooting

### 1. **Missing Area Totals**
**Issue**: Some areas show zero totals despite having customers with debt  
**Cause**: Customers not properly assigned to areas or area relationships missing

**Debug**:
```sql
-- Check for customers without area assignments
SELECT clientid, clientname, clientarea, clientdebt 
FROM client 
WHERE clientarea IS NULL OR clientarea = 0;

-- Verify government-area relationships
SELECT g.governmentname, ca.name as area_name, ga.id
FROM government g
JOIN goverarea ga ON g.governmetid = ga.governmentid
JOIN clientarea ca ON ga.clientareaid = ca.id;
```

### 2. **Incorrect Date Filtering**
**Issue**: Reports show no data despite valid transactions in date range  
**Cause**: Date field selection mismatch or timezone issues

**Debug**:
```sql
-- Check which date field contains data
SELECT COUNT(*) as sys_date_count, 
       MIN(clientdebtchangedate) as earliest_sys, 
       MAX(clientdebtchangedate) as latest_sys,
       COUNT(CASE WHEN userSelectedDate IS NOT NULL THEN 1 END) as user_date_count
FROM clientdebtchange 
WHERE clientid = [CLIENTID];
```

### 3. **Government Filter Not Working**
**Issue**: Government selection doesn't filter results properly  
**Cause**: Missing or incorrect goverarea table relationships

**Fix**:
```sql
-- Verify government-area mapping is complete
SELECT g.governmentname, COUNT(ga.clientareaid) as area_count
FROM government g
LEFT JOIN goverarea ga ON g.governmetid = ga.governmentid
GROUP BY g.governmetid, g.governmentname;

-- Check for orphaned areas
SELECT ca.name
FROM clientarea ca
LEFT JOIN goverarea ga ON ca.id = ga.clientareaid
WHERE ga.clientareaid IS NULL;
```

### 4. **Performance Issues with Large Areas**
**Issue**: Reports take too long to generate for large geographic areas  
**Cause**: Inefficient queries and lack of proper indexing

**Solutions**:
1. Add recommended indexes
2. Consider pagination for very large results
3. Implement caching for frequently accessed area reports
4. Optimize the aggregation queries

---

## 🧪 Testing Scenarios

### Test Case 1: Geographic Hierarchy Accuracy
```
1. Create test data with known government-area-client relationships
2. Add specific debt amounts to test clients
3. Run reports for different government/area combinations
4. Verify totals match manual calculations
5. Test edge cases (clients without areas, empty governments)
```

### Test Case 2: Date Field Selection
```
1. Create test transactions with different system and user dates
2. Run same report with both date field options
3. Verify results change appropriately based on date field selection
4. Test edge cases (null user dates, timezone boundaries)
```

### Test Case 3: Debit-Only Filtering
```
1. Create mix of customers with positive debt, zero debt, and negative debt
2. Run report with showDebit=1
3. Verify only customers with positive debt appear
4. Test with showDebit=0 to ensure all customers appear
```

### Test Case 4: Aggregation Accuracy
```
1. Manual calculation of area totals
2. Compare with report output
3. Verify government totals equal sum of area totals within government
4. Verify grand totals equal sum of all government totals
```

---

## 📚 Related Documentation

- [CLAUDE.md](/Applications/AMPPS/www/erp19/CLAUDE.md) - PHP 8.2 migration guide
- [clientReportsController.md](clientReportsController.md) - Individual customer reports
- [sellbillController.md](sellbillController.md) - Sales operations
- [clientController.php](#) - Customer management
- [Database Schema Documentation](#) - Geographic table relationships

---

**Documented By**: AI Assistant  
**Review Status**: ✅ Complete  
**Next Review**: When geographic structure changes occur