# Contact Us Controller Documentation

**File**: `/controllers/contactusController.php`  
**Purpose**: Manages customer contact form submissions with administrative response tracking and AJAX functionality  
**Last Updated**: December 20, 2024  
**Total Functions**: 4  
**Lines of Code**: ~219

---

## 📋 Overview

The Contact Us Controller manages customer inquiries and contact form submissions. It provides administrative functionality to view, respond to, and track contact form messages with user management and status tracking capabilities.

### Primary Functions
- [x] Contact form message management
- [x] Administrative response tracking
- [x] AJAX-powered DataTable interface
- [x] Message status management (read/unread)
- [x] Client vs visitor identification
- [x] Soft delete functionality
- [x] Advanced search and filtering

---

## 🗄️ Database Tables

| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **contactus** | Contact messages | id, clientid, name, mobile, email, subject, message, isRead, sysDate, userreadid, isdel |
| **user** | System users | userid, employeename |

---

## 🔑 Key Functions

### 1. **Default Action** - Display Contact Messages
**Location**: Lines 7-10  
**Purpose**: Show contact form management interface

### 2. **details** - View Message Details
**Location**: Lines 11-23  
**Purpose**: View and mark contact message as read

**Process Flow**:
1. Load message by ID
2. Mark as read and assign to current user
3. Update read status and date
4. Display message details with user information

### 3. **showajax** - AJAX DataTable Interface
**Location**: Lines 67-194  
**Purpose**: Provide paginated, searchable contact message interface

**Features**:
- **Advanced Filtering**: By date range, status, client type, user
- **Search Functionality**: Multi-column search capability
- **Pagination**: Server-side pagination for large datasets
- **Sorting**: Column-based sorting
- **Status Display**: Visual indicators for read/unread status

**Filter Parameters**:
```php
$start_date = $_POST['start_date'];
$end_date   = $_POST['end_date'];
$isdel      = $_POST['isdel'];        // Deleted status
$isRead     = $_POST['isRead'];       // Read status  
$id         = $_POST['id'];           // Specific message ID
$clientid   = $_POST['clientid'];     // Client filter
$userreadid = $_POST['userreadid'];   // User who read filter
```

**DataTable Response**:
```php
$output = array(
    "draw" => intval($_POST["draw"]),
    "recordsTotal" => intval($apps),
    "recordsFiltered" => $totals,
    "data" => array()
);
```

### 4. **removecontroller** - Soft Delete Message
**Location**: Lines 199-214  
**Purpose**: Mark message as deleted (soft delete)

**Process Flow**:
1. Load message by ID
2. Set deletion flag and metadata
3. Record deletion user and timestamp
4. Return success/failure status

---

## 🌐 URL Routes & Actions

| URL Parameter | Function | Description |
|---------------|----------|-------------|
| `do=` (empty) | Default | Show contact messages interface |
| `do=details` | View details | View specific message and mark as read |
| `do=getselectjoin` | AJAX helper | Generic select lookup functionality |
| `do=showajax` | AJAX data | DataTable data source |
| `do=removecontroller` | Delete | Soft delete message |

### DataTable Parameters
- **Search**: Global search across all columns
- **Pagination**: `start`, `length`, `draw` parameters
- **Sorting**: `order[0][column]`, `order[0][dir]`
- **Filtering**: Date range, status, client type filters

---

## 🧮 Query Logic

### Advanced Search Query Building
```php
$searchQuery = " ";

// Date range filtering
if($start_date != '' && $end_date != ''){
   $searchQuery .='and contactus.sysDate >= "' . $start_date . '" and contactus.sysDate <= "' . $end_date . '" ';
}

// Status filtering
if($isRead == 1){
   $searchQuery .= " and contactus.isRead > 0 "; 
}else{
   $searchQuery .= " and contactus.isRead = 0 "; 
}

// Multi-column search
if (isset($_POST['search']['value']) && $_POST['search']['value'] != "") {
    $searchQuery .= "and ( contactus.id LIKE '%" . $_POST["search"]["value"] . "%' 
                    OR contactus.name LIKE '%" . $_POST["search"]["value"] . "%'
                    OR contactus.email LIKE '%" . $_POST["search"]["value"] . "%'
                    OR contactus.subject LIKE '%" . $_POST["search"]["value"] . "%'
        )";
}
```

### Join Query for User Information
```sql
SELECT contactus.*, user.employeename  
FROM `contactus` 
LEFT JOIN user ON contactus.userreadid = user.userid
WHERE 1 {searchQuery}
```

---

## 🔒 Security & Performance

### Input Validation
- All inputs filtered via `filter_input()` functions
- SQL injection prevention through parameterized queries
- XSS prevention in output display

### Performance Considerations
- **Server-side pagination**: Handles large message volumes
- **Indexed searches**: Date and status fields should be indexed
- **Soft delete**: Maintains data integrity while hiding deleted records

### Required Indexes
```sql
CREATE INDEX idx_contactus_date ON contactus(sysDate);
CREATE INDEX idx_contactus_status ON contactus(isRead, isdel);
CREATE INDEX idx_contactus_user ON contactus(userreadid);
```

---

## 📊 Status Management

### Message Status Types
- **isRead = 0**: Unread message (default)
- **isRead = 1**: Read message  
- **isdel = 0**: Active message (default)
- **isdel = 1**: Deleted message

### Client Type Classification
- **clientid < 1**: Visitor (non-registered user)
- **clientid > 0**: Registered client

### Display Logic
```php
// Client type display
if($row["clientid"] < 1){
   $clientid = 'زائر';  // Visitor
}else{
   $clientid = 'عميل';  // Client
}

// Read status display
if($row["isRead"] < 1){
   $isRead = 'غير مقروءة';  // Unread
}else{
   $isRead = 'مقروءة';      // Read
}
```

---

## 📚 Related Documentation

- [CLAUDE.md](/Applications/AMPPS/www/erp19/CLAUDE.md) - PHP 8.2 migration guide
- [clientController.md](clientController.md) - Customer management
- [userController.md](userController.md) - User management

---

**Documented By**: AI Assistant  
**Review Status**: ✅ Complete  
**Next Review**: When major changes occur