# Login Controller Documentation

**File**: `/controllers/login.php`  
**Purpose**: Main authentication and system initialization controller for ERP19 system  
**Last Updated**: December 20, 2024  
**Total Functions**: 15+  
**Lines of Code**: ~1,404

---

## 📋 Overview

The Login Controller is the core authentication and system initialization module for the ERP19 system. It handles user authentication, session management, system initialization, database setup, software licensing, and user access control. This controller is critical for system security and proper functioning.

### Primary Functions
- [x] User authentication and login validation
- [x] Session management and security
- [x] System initialization and database setup
- [x] Software licensing and activation
- [x] Device and network access control
- [x] Database backup operations
- [x] User permission management
- [x] Security question handling
- [x] MAC address validation
- [x] Multi-database support

### Related Controllers
- [logout.php](logout.md) - Session termination
- [loginfunction.php](loginfunction.md) - Login utilities
- [index.php](index.md) - Main dashboard
- [userController.php](#) - User management

---

## 🗄️ Database Tables

### Authentication Tables (Primary)
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **user** | User accounts and authentication | userid, username, password, usergroupid, loginip, lastactivetime, blockedByLimits |
| **usergroup** | User permission groups | usergroupid, usergroupname, startpage, level, viewclients, viewbills |
| **properties** | System permissions/features | propertyid, propertyname, propertyparent, propertydefault |
| **relusergroupproperties** | Group-property relationships | usergroupid, propertyid, propertyvalue |

### System Configuration Tables
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **programsettings** | System configuration | programsettingsid, allowMobileLogin, lanAllowType, wanAllowType, serial, serial2, serial3 |
| **maccontrol** | Device access control | maccontrolid, mac, devicetype |
| **newdbname** | Database list | newdbnameid, dbname |
| **languages** | Language settings | languageid, languagecode, languagename |

### Initialization Tables
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **menuurl** | System menu structure | menuurlid, menuname, menuurl, parentid |
| **accountstree** | Chart of accounts | accountstreeid, accountname, accountparent |
| **billproperty** | Bill properties | billpropertyid, billpropertyname |
| **billsettings** | Bill configuration | billsettingsid, billnameid, billpropertyid, billsettingsvalue |

### Licensing Tables
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **sellbill** | Sales tracking for limits | sellbillid, sellbillclientid, sellbilltotalbill |
| **youtubelink** | Tutorial links | youtubelinkid, title, url |
| **assetscat** | Asset categories | assetscatid, assetscatname |

---

## 🔑 Key Functions

### 1. **Main Authentication Logic** - Default Action
**Location**: Lines 314-649  
**Purpose**: Handle user login, system initialization, and session creation

**Process Flow**:
1. Check if user already logged in - redirect to start page
2. Validate form submission and credentials
3. Check software activation and licensing
4. Initialize system tables if needed
5. Authenticate user credentials
6. Handle failed attempts and security questions
7. Set session variables and redirect

**Key Features**:
- Session persistence check
- Multi-database support
- Failed login attempt tracking
- Security question fallback
- Device and network access control
- Automatic database backup on login

---

### 2. **checkSoftwareActivation()** - License Validation
**Location**: Lines 210 (encoded)  
**Purpose**: Verify software license and activation status

**Process Flow**:
1. Check if serial numbers exist in database
2. Validate serial against system UUID
3. Decrypt and validate license file
4. Check expiration date
5. Set license session variables
6. Handle user limits

**Security Features**:
- Base64 encoded for protection
- Hardware UUID validation
- Expiration date checking
- User limit enforcement

---

### 3. **getUsedMAC()** - Device Identification
**Location**: Lines 1004-1044  
**Purpose**: Identify device MAC address and network type

**Process Flow**:
1. Detect connection type (server, LAN, WAN)
2. Execute platform-specific commands
3. Parse MAC address from system output
4. Determine network type
5. Return MAC and network information

**Platform Support**:
- Windows: `getmac` command
- Linux/Mac: ARP table parsing
- Server: Direct MAC extraction

---

### 4. **insertUser()** - Default User Creation
**Location**: Lines 809-846  
**Purpose**: Create default admin user during installation

**User Properties**:
```php
username: "admin"
password: "manager" 
employeename: "admin"
usergroupid: 1
saveid: 1
userstoreid: 0
```

---

### 5. **insertUserGroup()** - Default Group Creation
**Location**: Lines 698-747  
**Purpose**: Create default admin user group with full permissions

**Group Settings**:
- Full menu access (hidemenu: 1)
- All bill permissions enabled
- Admin-level access rights
- Default start page: "index.php"

---

### 6. **run_sql_file()** - Database Initialization
**Location**: Lines 1124-1174  
**Purpose**: Execute SQL scripts for system initialization

**Process Flow**:
1. Connect to database with proper charset
2. Load SQL file content
3. Remove comments and parse commands
4. Execute each SQL statement
5. Track success/failure rates

**Used For**:
- Initial table creation
- Default data insertion
- System configuration setup

---

### 7. **backup_tables()** - Automatic Backup
**Location**: Lines 1221-1355  
**Purpose**: Create daily database backups on login

**Features**:
- Daily, weekly, monthly backup rotation
- Compressed backup support (gzip)
- Cross-platform compatibility
- Automatic old backup cleanup
- Configuration-based retention

---

### 8. **handleSyncUsers()** - Concurrent Login Control
**Location**: Lines 1094-1122  
**Purpose**: Manage concurrent user sessions

**Process Flow**:
1. Check license sync settings
2. Compare IP addresses and last activity
3. Calculate time since last use
4. Allow or block based on timeout
5. Update login status

---

## 🔄 Workflows

### Workflow 1: New User Login
```
┌─────────────────────────────────────────────────────────────┐
│                  START: User Access System                 │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  1. Device & Network Validation                            │
│     - Check MAC address and device type                    │
│     - Verify network access permissions                    │
│     - Apply access restrictions                            │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  2. Software License Validation                            │
│     - Check activation status                              │
│     - Verify serial numbers                                │
│     - Validate expiration date                             │
│     - Check user limits                                     │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  3. System Initialization                                  │
│     - Initialize database tables                           │
│     - Create default users/groups                          │
│     - Set up system properties                             │
│     - Configure bill settings                              │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  4. User Authentication                                     │
│     - Validate username/password                           │
│     - Check account status                                 │
│     - Handle failed attempts                               │
│     - Security question fallback                           │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  5. Session Management                                      │
│     - Create user session                                  │
│     - Set permissions and preferences                      │
│     - Handle concurrent logins                             │
│     - Redirect to start page                               │
└─────────────────────────────────────────────────────────────┘
```

---

## 🌐 URL Routes & Actions

| URL Parameter | Function Called | Description |
|---------------|----------------|-------------|
| `do=` (empty) | Default login flow | Check session or show login form |
| `do=en` | Authentication logic | Process login form submission |
| `do=enq` | Answer validation | Process security question answer |
| `do=finished` | License limit reached | Show user limit message |
| `do=activate` | License activation | Show activation form or process |
| `do=usernolimitation` | User limit error | Display user limitation message |
| `do=usednow` | Session conflict | Show "account in use" message |

### Required Parameters by Action

**Login Form** (`do=en`):
- `username` - User login name
- `password` - User password
- `db` - Database selection

**Security Question** (`do=enq`):
- `username` - User login name  
- `answer` - Security question answer

**License Activation** (`do=activate`):
- `serial` - Activation serial number

---

## 🔒 Security & Permissions

### Access Control Layers

**1. Network Level**:
```php
// Device type checking
$deviceType = checkDevice(); // DESKTOP, MOBILE, TABLET

// Network type validation  
list($mac, $networkType) = getUsedMAC(); // LAN, WAN, SERVER

// Access restrictions
if ($Programsetting->allowMobileLogin == 1 && $deviceType != "DESKTOP") {
    // Block mobile access
}
```

**2. MAC Address Control**:
```php
// Check allowed devices
$macData = $macControlEX->queryByMacAndDeviceType($mac, $deviceType);
if (count($macData) < 1) {
    // Block unregistered device
}
```

**3. License Validation**:
- Hardware UUID matching
- Serial number verification
- Expiration date checking
- User count limits

**4. Session Security**:
- Failed attempt tracking
- Security question fallback
- Concurrent login prevention
- IP address validation

---

## 📊 Performance Considerations

### Database Optimization Tips
1. **Connection Caching**: Single connection reuse
2. **Prepared Statements**: All SQL uses DAO layer parameterization  
3. **Efficient Queries**: Minimal data fetching during login
4. **Index Requirements**:
   - `user(username)`
   - `user(loginip, lastactivetime)`
   - `usergroup(usergroupid)`

### Known Performance Issues
```sql
-- Heavy table loading during initialization
-- Solution: Cache initialization status
SELECT COUNT(*) FROM properties; -- Can be slow on first run

-- Multiple file operations during backup
-- Solution: Background processing recommended
```

---

## 🐛 Common Issues & Troubleshooting

### 1. **"Software not activated" Error**
**Cause**: Missing or invalid license
**Fix**: 
1. Check license.txt file exists
2. Verify serial numbers in database
3. Re-activate if expired

### 2. **"User blocked by limits" Error**  
**Cause**: License user count exceeded
**Fix**:
1. Check current user count vs license
2. Deactivate unused users
3. Upgrade license

### 3. **MAC Address Access Denied**
**Cause**: Device not registered or wrong network
**Fix**:
1. Add MAC to maccontrol table
2. Check network settings
3. Verify device type detection

### 4. **Database Connection Failures**
**Cause**: Incorrect database configuration
**Fix**:
```php
// Check ConnectionProperty.class.php settings
$host = ConnectionProperty::getHost();
$user = ConnectionProperty::getUser();
$db = ConnectionProperty::getDatabase();
```

### 5. **Session Conflicts**
**Cause**: Same user from multiple devices
**Fix**:
1. Increase sync time in license
2. Allow multiple sessions
3. Clear stuck sessions

---

## 🧪 Testing Scenarios

### Test Case 1: First-Time Installation
```
1. Access system with empty database
2. Verify automatic table creation
3. Check default user/group creation  
4. Confirm successful admin login
```

### Test Case 2: License Expiration
```
1. Set license expiration to past date
2. Attempt login
3. Verify redirect to activation page
4. Test re-activation process
```

### Test Case 3: Device Access Control
```
1. Enable MAC address control
2. Login from unregistered device
3. Verify access denial
4. Add device and retry
```

### Test Case 4: Failed Login Attempts
```
1. Enter wrong password 3 times
2. Verify security question prompt
3. Test correct answer flow
4. Test password reset
```

---

## 📚 Related Documentation

- [CLAUDE.md](/Applications/AMPPS/www/erp19/CLAUDE.md) - PHP 8.2 migration fixes
- [logout.php](logout.md) - Session termination
- [authentication.php](#) - Permission checking
- [impOpreation.php](#) - Core system functions

---

**Documented By**: AI Assistant  
**Review Status**: ✅ Complete  
**Next Review**: When major security changes occur