# Login Function Controller Documentation

**File**: `/controllers/loginfunction.php`  
**Purpose**: Login utilities and system initialization helper functions  
**Last Updated**: December 20, 2024  
**Total Functions**: 6+  
**Lines of Code**: ~274

---

## 📋 Overview

The Login Function Controller serves as a utility module for system initialization and user group management. It handles:
- System properties initialization
- Default user group creation
- User account setup
- Database schema setup
- SQL file execution
- System property management

### Primary Functions
- [x] System properties initialization from SQL files
- [x] Default admin user group creation
- [x] Default admin user account creation
- [x] Database initialization
- [x] SQL file processing
- [x] User group permissions setup

### Related Controllers
- [login.php](login.md) - Main authentication controller
- [userController.php](userController.md) - User management
- [usergroupController.php](usergroupController.md) - User group management

---

## 🗄️ Database Tables

### Primary Tables (Direct Operations)
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **properties** | System properties/permissions | propertyid, propertyname, propertyparent, propertydefault |
| **usergroup** | User groups/roles | usergroupid, usergroupname, startpage, level |
| **user** | User accounts | userid, username, password, usergroupid |
| **relusergroupproperties** | User group permissions | usergroupid, propertyid, propertyvalue |

---

## 🔑 Key Functions

### 1. **Default Action** - Property Initialization
**Location**: Line 85  
**Purpose**: Initialize system properties and redirect to login

**Process Flow**:
1. Execute properties SQL file
2. Create default user group
3. Redirect to login page

**Features**:
- SQL file execution
- User group creation
- System initialization

---

### 2. **addUserGroup()** - Create Default Admin Group
**Location**: Line 115  
**Purpose**: Create default admin user group with full permissions

**Function Signature**:
```php
function addUserGroup() {
    global $myUsergroup, $myUsergroupRecord, $myUsergroupEx;
}
```

**Process Flow**:
1. Check if admin group exists
2. Set group properties:
   - Name: "admin"
   - Start page: "articleController.php"
   - Level: 1 (admin level)
   - Menu visibility: enabled
   - Various permissions
3. Insert user group
4. Add group permissions
5. Create admin user account

**Group Permissions Set**:
- `hidemenu = 1` - Show menu
- `hidecat = 0` - Show categories
- `clientnegative = 1` - Allow negative client balances
- `storenegative = 1` - Allow negative store balances
- Various bill and pricing permissions

---

### 3. **addReluesUsergroup()** - Setup Group Permissions
**Location**: Line 184  
**Purpose**: Add property permissions for user group

**Function Signature**:
```php
function addReluesUsergroup($usergroupId) {
    // Add all system properties to group with default values
}
```

**Process Flow**:
1. Load all system properties
2. For admin group (ID = 1):
   - Add each property with value 0 (allowed)
   - Create permission records

---

### 4. **addUser()** - Create Default Admin User
**Location**: Line 207  
**Purpose**: Create default admin user account

**Function Signature**:
```php
function addUser($usergroupId) {
    global $myUser, $myUserRecord;
}
```

**Process Flow**:
1. Check if user exists for group
2. Set user properties:
   - Username: "admin"
   - Password: "manager"
   - User level: matches group ID
3. Insert user record

---

### 5. **run_sql_file()** - SQL File Processor
**Location**: Line 225  
**Purpose**: Execute SQL commands from file

**Function Signature**:
```php
function run_sql_file($location) {
    // Process and execute SQL file
}
```

**Process Flow**:
1. Connect to database (hardcoded MySQL connection)
2. Set charset to UTF-8
3. Insert hardcoded property record
4. Load SQL file content
5. Remove comments
6. Split into individual commands
7. Execute each command
8. Return success/total counts

**Legacy Code Issues**:
- Uses deprecated `mysql_*` functions
- Hardcoded database credentials
- Mixed encoding handling
- No error handling

---

## 🔄 Workflows

### Workflow 1: System Initialization
```
┌─────────────────────────────────────────────────────────────┐
│                START: System Setup                         │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  1. Initialize Properties                                   │
│     - Execute pro.sql file                                  │
│     - Load system properties                                │
│     - Setup permission structure                            │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  2. Create Admin User Group                                 │
│     - Check if admin group exists                           │
│     - Set group properties and permissions                  │
│     - Insert user group record                              │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  3. Setup Group Permissions                                 │
│     - Load all system properties                            │
│     - Create permission records for admin group             │
│     - Set default permission values                         │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  4. Create Admin User                                       │
│     - Check if admin user exists                            │
│     - Set user credentials (admin/manager)                  │
│     - Insert user record                                    │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  5. Redirect to Login                                       │
│     - System ready for use                                  │
│     - Redirect to login.php                                 │
└─────────────────────────────────────────────────────────────┘
```

---

## 🌐 URL Routes & Actions

| URL Parameter | Function Called | Description |
|---------------|----------------|-------------|
| `do=` (empty) | Default action | Initialize system properties and setup |
| `do=sucess` | Success page | Display success message |
| `do=error` | Error page | Display error message |

---

## 🔒 Security & Permissions

### Default Admin Setup
```php
// Default admin credentials
$myUser->username = 'admin';
$myUser->password = 'manager';  // Plain text - SECURITY RISK
$myUser->userlevel = $usergroupId;
```

**Security Issues**:
- Plain text password storage
- Hardcoded credentials
- No password complexity requirements
- No account lockout mechanisms

### Permission Structure
```php
// Default permission setup
$myRelusergrouppropertie->propertyvalue = 0;  // 0 = allowed, 1 = denied
```

---

## 📊 Performance Considerations

### Database Issues
1. **Deprecated MySQL Extension**: Uses old `mysql_*` functions
2. **Hardcoded Connection**: Direct database credentials
3. **No Connection Pooling**: Creates new connections
4. **No Query Optimization**: Basic SQL execution

### Legacy Code Problems
```php
// DEPRECATED: mysql_* functions
$con = mysql_connect("localhost", "root", "123456");
mysql_select_db("arabcity", $con);

// Should use MySQLi or PDO instead
```

---

## 🐛 Common Issues & Troubleshooting

### 1. **Database Connection Failures**
**Issue**: Cannot connect to MySQL  
**Cause**: Hardcoded credentials or deprecated MySQL extension

**Debug**:
```php
// Check MySQL extension
if (!extension_loaded('mysql')) {
    echo "MySQL extension not loaded";
}

// Check connection
$con = mysql_connect("localhost", "root", "123456");
if (!$con) {
    echo "Connection failed: " . mysql_error();
}
```

### 2. **Property Initialization Errors**
**Issue**: Properties not loaded correctly  
**Cause**: SQL file execution failure

**Debug**:
```php
// Check SQL file existence
if (!file_exists("initialValues/pro.sql")) {
    echo "SQL file not found";
}

// Check file permissions
if (!is_readable("initialValues/pro.sql")) {
    echo "SQL file not readable";
}
```

### 3. **User Group Creation Errors**
**Issue**: Admin group not created  
**Cause**: Variable scope or database errors

**Fix**:
```php
// Check variable declaration
global $myUsergroup, $myUsergroupRecord;

// Verify group creation
$usergroupValidation = $myUsergroupEx->queryAllWithName("admin");
if (count($usergroupValidation) <= 0) {
    echo "Admin group creation failed";
}
```

### 4. **Permission Setup Issues**
**Issue**: Permissions not assigned correctly  
**Cause**: Property loading or insertion failures

**Debug**:
```sql
-- Check properties loaded
SELECT COUNT(*) FROM properties;

-- Check permissions created
SELECT COUNT(*) FROM relusergroupproperties WHERE usergroupid = 1;
```

---

## 🧪 Testing Scenarios

### Test Case 1: Fresh Installation
```
1. Clear database tables
2. Access loginfunction.php
3. Verify properties loaded
4. Check admin group created
5. Confirm admin user exists
6. Test login with admin/manager
```

### Test Case 2: Duplicate Initialization
```
1. Run initialization twice
2. Verify no duplicate groups
3. Check no duplicate users
4. Confirm permissions intact
```

### Test Case 3: SQL File Processing
```
1. Test with missing SQL file
2. Test with corrupted SQL file
3. Verify error handling
4. Check rollback behavior
```

---

## ⚠️ Upgrade Requirements

### Immediate Security Fixes Needed
1. **Password Hashing**: Implement proper password encryption
2. **Database Layer**: Replace deprecated MySQL functions
3. **Error Handling**: Add proper exception handling
4. **Input Validation**: Add parameter sanitization

### Recommended Improvements
```php
// Replace deprecated MySQL with MySQLi
$conn = new mysqli($host, $user, $password, $database);

// Add password hashing
$hashedPassword = password_hash('manager', PASSWORD_DEFAULT);

// Add proper error handling
try {
    $result = $conn->query($sql);
    if (!$result) {
        throw new Exception($conn->error);
    }
} catch (Exception $e) {
    error_log($e->getMessage());
}
```

---

## 📚 Related Documentation

- [CLAUDE.md](/Applications/AMPPS/www/erp19/CLAUDE.md) - PHP 8.2 migration guide
- [login.php](login.md) - Main authentication controller
- [userController.php](userController.md) - User management
- [usergroupController.php](usergroupController.md) - User group management

---

**Documented By**: AI Assistant  
**Review Status**: ✅ Complete  
**Security Review**: ❌ Critical issues identified  
**Next Review**: Immediate security updates required