# Notices Controller Documentation

**File**: `/controllers/noticesController.php`  
**Purpose**: System notifications and alerts management for inventory, bills, checks, and operational status  
**Last Updated**: December 20, 2024  
**Total Functions**: 15+  
**Lines of Code**: ~817

---

## 📋 Overview

The Notices Controller provides comprehensive business notifications including low inventory alerts, bill due dates, check processing status, workshop delays, and insurance approvals. It serves as a central dashboard for monitoring critical business operations requiring attention.

### Primary Functions
- [x] Inventory level monitoring and alerts
- [x] Product sales tracking and non-selling product identification
- [x] Check processing notifications (clients and suppliers)
- [x] Bill payment due date alerts
- [x] Workshop order delay notifications
- [x] Insurance bill approval status
- [x] Premium payment reminders
- [x] Product request notifications
- [x] Category-based filtering and reporting

---

## 🗄️ Database Tables

### Core Notification Tables
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **product** | Product master data | productid, productName, productCatId |
| **storedetail** | Inventory quantities | storedetailid, productid, storedetailquantity, storeid |
| **sellbilldetail** | Sales transaction details | sellbilldetailproductid, sellbilldetaildate |
| **datedchecked** | Post-dated checks | datedcheckedid, checkvalue, clientnum, addtype, done |
| **kempiala** | Promissory notes | kempialaid, kempialadate, kempialavalue |

### Workshop & Insurance Tables
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **toworkshoporder** | Workshop orders | toworkshoporderid, deliverworkshopdate, recieveworkshopdate, orderstatus |
| **bills** | Service bills | billid, clientid, insurance, approved, notifseen |
| **premium** | Premium payments | premiumid, premiumdate, userid |

---

## 🔑 Key Functions

### 1. **Default Dashboard** - System Notifications Overview
**Location**: Lines 231-291  
**Purpose**: Display comprehensive business notification dashboard

**Process Flow**:
1. **Category Data**: Load product categories for filtering
2. **Store Loading**: Load warehouse/store information  
3. **Product Analysis**: Get low stock and non-selling products
4. **Bill Analysis**: Check overdue bills and payments
5. **Check Processing**: Monitor dated checks requiring attention
6. **Workshop Orders**: Track delayed workshop operations
7. **Insurance Bills**: Monitor insurance approval status
8. **Premium Tracking**: Check due premium payments

### 2. **showallproductbystorename()** - Store-Specific Product Analysis
**Location**: Lines 480-510  
**Purpose**: Analyze product levels filtered by store and category

**Features**:
- Store-specific filtering
- Category hierarchy support
- Product path generation for display
- Quantity threshold monitoring

### 3. **productnotsellbystorename()** - Non-Selling Product Detection
**Location**: Lines 546-582  
**Purpose**: Identify products with no sales in past 7 days by store

### 4. **getToWorkshopOrderNoticeInDetails()** - Workshop Delay Monitoring  
**Location**: Lines 658-711  
**Purpose**: Monitor workshop order delays across different stages

**Delay Types Monitored**:
- **Delivery Delays**: Orders not delivered to workshop on time
- **Workshop Delays**: Orders not completed by workshop deadline  
- **Return Delays**: Completed orders not returned to branch on time

### 5. **getInsuranceBillNotif()** - Insurance Processing Alerts
**Location**: Lines 723-793  
**Purpose**: Track insurance bill approval status and notifications

**Status Types**:
- **Waiting Bills**: Pending approval (`approved = 0`)
- **Approved Bills**: Recently approved (`approved = 1`)
- **Rejected Bills**: Recently rejected (`approved = 2`)

### 6. **getPremiumTodays()** - Premium Due Notifications
**Location**: Lines 796-805  
**Purpose**: Get premium payments due today for current user

---

## 🔄 Workflows

### Notification Dashboard Workflow
```
┌─────────────────────────────────────────────────────────────┐
│            START: Load Notifications Dashboard             │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  1. Load Basic Data                                        │
│     ├─→ Load product categories                            │
│     ├─→ Load store/warehouse information                   │
│     └─→ Load user permissions and settings                 │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  2. Analyze Inventory Status                               │
│     ├─→ showallproductx2() - Low stock products           │
│     ├─→ productnotsell() - Non-selling products (7 days)  │
│     └─→ Apply category and store filters                  │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  3. Check Financial Obligations                           │
│     ├─→ showallkempila() - Due promissory notes           │
│     ├─→ checkforclient() - Client checks due (7 days)     │
│     └─→ checkforsupplier() - Supplier checks due (7 days) │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  4. Monitor Operational Issues                             │
│     ├─→ getToWorkshopOrderNoticeInDetails() - Delays      │
│     ├─→ getInsuranceBillNotif() - Insurance status        │
│     ├─→ getPremiumTodays() - Due premiums                 │
│     └─→ getProductRequest() - Product requests            │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  5. Display Notifications Dashboard                       │
│     └─→ noticesview/show.html with all alert counts       │
└─────────────────────────────────────────────────────────────┘
```

---

## 🌐 URL Routes & Actions

| URL Parameter | Function | Description |
|---------------|----------|-------------|
| `do=` (empty) | Default | Main notifications dashboard |
| `do=show` | Store filter | Filter by specific store |
| `do=showproductnotsell` | Non-selling | Show non-selling products by store |
| `do=editprint` | Bill print | Print promissory note details |
| `do=editprint2` | Check print | Print check details |

### Filter Parameters
- `supplierid` - Store/supplier filter ID
- `level` - Category level for hierarchical filtering  
- `productCatId{N}` - Category ID for level N filtering

---

## 🧮 Calculation Methods

### Time-Based Calculations
```php
// Workshop delay calculation
function dateTimeDiff($dateTime1, $dateTime2) {
    $dateTime1 = new DateTime($dateTime1);
    $dateTime2 = new DateTime($dateTime2);
    $diff = $dateTime2->diff($dateTime1);
    $hours = $diff->h + ($diff->days * 24);
    return $hours;
}
```

### Product Path Generation
```php
// Recursive category path building
$parentId = $product->productCatId;
$pathArr = getProductPath_recursive1($parentId, $categories, 0);
$request->productName = $product->productName . '/' . $pathArr;
```

---

## 🔒 Security & Performance

### Performance Considerations
- **Multiple Query Execution**: Dashboard runs many analysis queries
- **Date Range Calculations**: 7-day lookbacks on transaction tables
- **Recursive Path Building**: Category hierarchy traversal
- **User Permission Filtering**: Role-based data access

### Required Indexes
- `sellbilldetail(sellbilldetailproductid, sellbilldetaildate)`
- `datedchecked(checkdate, addtype, done)`
- `toworkshoporder(orderstatus, deliverworkshopdate)`
- `bills(insurance, approved, notifseen)`

---

## 📚 Related Documentation

- [CLAUDE.md](/Applications/AMPPS/www/erp19/CLAUDE.md) - PHP 8.2 migration guide
- [noticefunctions.md](noticefunctions.md) - Notice utility functions
- [productController.md](productController.md) - Product management

---

**Documented By**: AI Assistant  
**Review Status**: ✅ Complete  
**Next Review**: When major changes occur