# Payed Controller Documentation

**File**: `/controllers/payedController.php`  
**Purpose**: Simple view controller for payment status display pages  
**Last Updated**: December 20, 2024  
**Total Functions**: 0 (View-only controller)  
**Lines of Code**: ~66

---

## 📋 Overview

The Payed Controller is a minimal view controller that serves static payment status pages. It does not contain business logic or data processing functions, instead focusing solely on displaying payment-related views through Smarty templates.

### Primary Functions
- [x] Display "payed" (paid) status page
- [x] Display "notpayed" (unpaid) status page  
- [x] Display error page
- [x] Basic authentication integration
- [x] Template rendering via Smarty

### Related Controllers
- [clientPayedDeptController.php](clientPayedDeptController.md) - Payment processing
- [clientPayedDeptReportController.php](clientPayedDeptReportController.md) - Payment reporting
- [paymentMethodsController.php](paymentMethodsController.md) - Payment methods management

---

## 🗄️ Database Tables

### Primary Tables (Indirect Access via Templates)
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **programsettings** | System configuration | programsettingsid, settingkey, settingvalue |

### Notes
- This controller does not directly interact with database tables
- Database access occurs through included templates and authentication system
- Program settings may be referenced by included templates

---

## 🔑 Key Functions

### 1. **Default Action** - Payed Status Display
**Location**: Lines 44-49  
**Purpose**: Display paid status confirmation page

**Function Signature**:
```php
if ($do == "payed" || empty($do)) {
    include_once("../public/authentication.php");
    $smarty->display("payedview/payed.html");
}
```

**Process Flow**:
1. Check authentication status
2. Load authentication system
3. Display payed confirmation template

---

### 2. **Not Payed Action** - Unpaid Status Display  
**Location**: Lines 50-55  
**Purpose**: Display unpaid status notification page

**Function Signature**:
```php
elseif ($do == "notpayed") {
    include_once("../public/authentication.php");
    $smarty->display("payedview/notpayed.html");
}
```

**Process Flow**:
1. Check authentication status
2. Load authentication system
3. Display unpaid notification template

---

### 3. **Error Action** - Error Display
**Location**: Lines 56-60  
**Purpose**: Display error page for payment-related issues

**Function Signature**:
```php
elseif ($do == "error") {
    $smarty->display("error.html");
}
```

**Process Flow**:
1. Display generic error template
2. No authentication required for error display

---

## 🔄 Workflows

### Workflow 1: Payment Status Display
```
┌─────────────────────────────────────────────────────────────┐
│              START: Access Payment Status Page             │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  1. Parse URL Parameter 'do'                               │
│     - Check for 'payed', 'notpayed', or 'error'           │
│     - Default to 'payed' if empty                         │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  2. Authentication Check (if needed)                       │
│     - Include authentication.php for payed/notpayed       │
│     - Skip authentication for error pages                 │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  3. Display Appropriate Template                           │
│     - payedview/payed.html for success                    │
│     - payedview/notpayed.html for pending                 │
│     - error.html for errors                               │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  4. Render Footer Template                                 │
│     - Display footer.html in all cases                    │
└─────────────────────────────────────────────────────────────┘
```

---

## 🌐 URL Routes & Actions

| URL Parameter | Template Displayed | Authentication Required | Description |
|---------------|-------------------|------------------------|-------------|
| `do=payed` or empty | `payedview/payed.html` | Yes | Payment successful confirmation |
| `do=notpayed` | `payedview/notpayed.html` | Yes | Payment pending notification |
| `do=error` | `error.html` | No | Error message display |

### URL Examples
- `/controllers/payedController.php` - Shows paid status (default)
- `/controllers/payedController.php?do=payed` - Shows paid status
- `/controllers/payedController.php?do=notpayed` - Shows unpaid status  
- `/controllers/payedController.php?do=error` - Shows error page

---

## 🧮 Calculation Methods

**No calculations performed** - This controller is purely for view presentation.

---

## 🔒 Security & Permissions

### Authentication Requirements
```php
// Required for payment status pages
include_once("../public/authentication.php");
```

**Security Features**:
- Authentication required for payed/notpayed views
- Error pages accessible without authentication
- Uses standard ERP authentication system
- No direct user input processing (minimal attack surface)

### Input Validation
- Only accepts predefined `$_GET['do']` values
- No user data processing or SQL operations
- Minimal security risk due to static nature

---

## 📊 Performance Considerations

### Optimization Strengths
1. **Minimal Processing**: No database operations or complex logic
2. **Fast Loading**: Simple template rendering only
3. **Low Memory Usage**: No data arrays or complex objects
4. **Cacheable**: Static content can be easily cached

### Performance Characteristics
- **Load Time**: <50ms (template rendering only)
- **Memory Usage**: Minimal (authentication + templates)
- **Database Queries**: 0 direct queries
- **File I/O**: Template files only

---

## 🐛 Common Issues & Troubleshooting

### 1. **Template Not Found**
**Issue**: "Template not found" error for payment views  
**Cause**: Missing template files in `/templates/payedview/` directory

**Fix**:
```bash
# Check template existence
ls -la /templates/payedview/
# Should contain: payed.html, notpayed.html
```

### 2. **Authentication Failure**
**Issue**: Redirected to login instead of payment status  
**Cause**: Session expired or authentication.php issues

**Debug**:
```php
// Add before authentication include
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
echo "Session ID: " . session_id();
echo "User ID: " . $_SESSION['userid'];
```

### 3. **Footer Not Displaying**
**Issue**: Page cuts off without footer  
**Cause**: Error in template rendering or includes

**Fix**:
```php
// Verify template path
if (file_exists("../templates/footer.html")) {
    echo "Footer template exists";
} else {
    echo "Footer template missing";
}
```

---

## 🧪 Testing Scenarios

### Test Case 1: Payment Success Flow
```
1. Navigate to /controllers/payedController.php
2. Verify authentication prompt if not logged in
3. Confirm payed.html template displays correctly
4. Check footer renders properly
```

### Test Case 2: Payment Pending Flow
```
1. Navigate to /controllers/payedController.php?do=notpayed
2. Verify authentication prompt if not logged in
3. Confirm notpayed.html template displays correctly
4. Check for proper messaging and styling
```

### Test Case 3: Error Handling
```
1. Navigate to /controllers/payedController.php?do=error
2. Verify NO authentication prompt (direct access)
3. Confirm error.html template displays
4. Check error styling and messaging
```

### Test Case 4: Invalid Parameter
```
1. Navigate to /controllers/payedController.php?do=invalid
2. Should default to payed.html template
3. Verify authentication still required
```

---

## 📚 Related Documentation

- [CLAUDE.md](/Applications/AMPPS/www/erp19/CLAUDE.md) - PHP 8.2 migration guide
- [clientPayedDeptController.md](clientPayedDeptController.md) - Payment processing
- [paymentMethodsController.md](paymentMethodsController.md) - Payment methods
- [Authentication System Documentation](#) - Login/session management
- [Smarty Templates Guide](#) - Template system reference

---

**Documented By**: AI Assistant  
**Review Status**: ✅ Complete  
**Next Review**: When template changes occur