# Product Receive Controller Documentation

**File**: `/controllers/productRecieveController.php`  
**Purpose**: Manages maintenance product receipt operations with spare parts tracking  
**Last Updated**: December 20, 2024  
**Total Functions**: 5+  
**Lines of Code**: ~272

---

## 📋 Overview

The Product Receive Controller handles maintenance operations for spare parts reception. It manages:
- Spare parts reception from clients
- Inventory quantity updates
- Store report generation
- Receipt tracking and pagination
- Client and spare part management

### Primary Functions
- [x] Add spare parts receipts
- [x] Update spare parts inventory
- [x] Generate store movement reports
- [x] Display receipt history with pagination
- [x] Filter by spare part type

---

## 🗄️ Database Tables

### Primary Tables
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **productrecieve** | Receipt records | productrecieveid, clientname, sparepartid, cost, partsnumber, recieptnum, type |
| **sparepartstoredetail** | Spare parts inventory | sparepartstoredetailid, sparepartid, partquantity, sparepartstoreId |
| **sparepartstorereport** | Movement tracking | storereportid, sparepartid, partquantity, partbefore, partafter, processname |
| **sparepart** | Spare parts master | sparepartid, sparepartname, conditions |

---

## 🔑 Key Functions

### 1. **add()** - Create Receipt
**Location**: Line 155  
**Purpose**: Process spare parts receipt and update inventory

**Process Flow**:
1. **Receipt Creation**:
   ```php
   $productRecieve->clientname = $clientname;
   $productRecieve->cost = $cost;
   $productRecieve->partsnumber = $partsnumber;
   $productRecieve->sparepartid = $sparepartid;
   $productRecieveId = $productRecieveDAO->insert($productRecieve);
   ```

2. **Inventory Update**:
   ```php
   $partsDetail = $sparePartDetailExt->queryBySparepartidExt($sparepartid);
   $sparePartDetail->partquantity = $partsDetail->partquantity - $partsnumber;
   $sparePartDetailExt->updateQuantity($sparePartDetail);
   ```

3. **Movement Tracking**:
   ```php
   $sparePartReport->partafter = $partsDetail->partquantity - $partsnumber;
   $sparePartReport->partbefore = $partsDetail->partquantity;
   $sparePartReport->processname = "استلام منتج";
   $sparePartReportDAO->insert($sparePartReport);
   ```

### 2. **show() / showWithSparePartId()** - Display Receipts
**Location**: Lines 210, 242  
**Purpose**: Display receipt history with pagination and filtering

**Features**:
- Pagination support (25 records per page)
- Spare part filtering
- Arabic navigation text
- Extended query capabilities

---

## 🔄 Workflows

### Workflow: Spare Parts Reception
```
┌─────────────────────────────────────────┐
│     START: Receive Spare Part          │
└─────────────┬───────────────────────────┘
              │
              ▼
┌─────────────────────────────────────────┐
│  1. Create Receipt Record               │
│     - Client information                │
│     - Cost and quantity                 │
│     - Receipt number                    │
└─────────────┬───────────────────────────┘
              │
              ▼
┌─────────────────────────────────────────┐
│  2. Update Inventory                    │
│     - Decrease available quantity       │
│     - Update store details             │
└─────────────┬───────────────────────────┘
              │
              ▼
┌─────────────────────────────────────────┐
│  3. Generate Movement Report           │
│     - Track before/after quantities    │
│     - Record process name               │
│     - Link to receipt record           │
└─────────────────────────────────────────┘
```

---

## 🌐 URL Routes & Actions

| URL Parameter | Function Called | Description |
|---------------|----------------|-------------|
| `do=` (empty) | Default form | Display receipt form |
| `do=add` | `add()` | Process receipt |
| `do=show` | `show()` / `showWithSparePartId()` | Display receipt history |

### Required Parameters
**Add Receipt** (`do=add`):
- `recieptnum` - Receipt number
- `clientname` - Client name
- `type` - Receipt type
- `sparepartid` - Spare part ID
- `cost` - Cost amount
- `partsnumber` - Quantity received

---

## 🧮 Calculation Methods

### Inventory Decrease
```php
$newQuantity = $currentQuantity - $receivedQuantity;
$sparePartDetail->partquantity = $partsDetail->partquantity - $partsnumber;
```

### Movement Tracking
```php
$sparePartReport->partafter = $partsDetail->partquantity - $partsnumber;
$sparePartReport->partbefore = $partsDetail->partquantity;
$sparePartReport->partquantity = $partsnumber; // Amount received
```

---

## 🔒 Security & Permissions

### Authentication
```php
include_once("../public/authentication.php");
```

### Input Validation
- Posted data extraction and type casting
- Session user ID tracking
- Database transaction integrity

---

## 📊 Performance Considerations

### Pagination Implementation
- 25 records per page limit
- Efficient counting queries
- Proper URL parameter handling

### Database Optimization
- Uses extended DAO classes for complex queries
- Proper transaction handling for inventory updates

---

## 🧪 Testing Scenarios

### Test Case 1: Receipt Processing
```
1. Create new receipt with valid data
2. Verify inventory decrease
3. Check movement report generation
4. Validate receipt record creation
```

### Test Case 2: Pagination
```
1. Add multiple receipts (>25)
2. Test pagination navigation
3. Verify record counts
4. Check filtering functionality
```

---

**Documented By**: AI Assistant  
**Review Status**: ✅ Complete  
**Next Review**: When major changes occur