# Product By Code Controller Documentation

**File**: `/controllers/productbycode.php`  
**Purpose**: Product lookup system using barcodes and internal product IDs  
**Last Updated**: December 20, 2024  
**Total Functions**: 2+  
**Lines of Code**: ~128

---

## 📋 Overview

The Product By Code Controller provides a search interface for products using barcodes or internal product IDs. Features:
- Barcode scanning and ID lookup
- Store quantity display
- Machine-friendly interface (no login required)
- Dual search modes (barcode vs ID)
- Program settings integration

### Primary Functions
- [x] Product search by barcode/ID
- [x] Store quantity checking
- [x] Dual search mode handling
- [x] No-login machine interface
- [x] Program settings integration

---

## 🗄️ Database Tables

### Primary Tables
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **product** | Product master data | productId, productName, productCatId |
| **storedetail** | Store quantities | storeid, productid, productquantity |
| **programsettings** | System configuration | programsettingsid, usedParcode, settingvalue |

---

## 🔑 Key Functions

### 1. **Default Action** - Search Interface
**Location**: Line 64  
**Purpose**: Display search form with program settings

**Process Flow**:
1. Load program settings
2. Apply store filtering based on session
3. Display search interface
4. Set machine interface flags

### 2. **getData()** - Product Lookup
**Location**: Line 105  
**Purpose**: Search and display product information

**Search Logic**:
```php
$text = str_replace(["ه", "÷", "/"], "i", $text);
$parcodePrefixList = array('i', 'I');

if (in_array($text[0], $parcodePrefixList) && $Programsettingdata->usedParcode == 1) {
    $type = "id";
    $trimmed = trim($text, "i,I,ه");
    $barcode = intval($trimmed);
} else {
    $type = "barcode";
    $barcode = $text;
}
```

**Display Logic**:
```php
if ($type == "id") {
    $productsData = $productExt->queryByProductCatIdInExts($barcode);
} else {
    $productsData = $productExt->queryByProductCatIdInExts2($barcode);
}

if (isset($productsData[0]->productId)) {
    $storesQuantity = $myStoredetailEx->getproductInStores($productsData[0]->productId);
}
```

---

## 🔄 Workflows

### Workflow: Product Search
```
┌─────────────────────────────────────────┐
│       START: Scan/Enter Code           │
└─────────────┬───────────────────────────┘
              │
              ▼
┌─────────────────────────────────────────┐
│  1. Code Processing                     │
│     - Clean input text                  │
│     - Detect barcode vs ID             │
│     - Apply prefix rules               │
└─────────────┬───────────────────────────┘
              │
              ▼
┌─────────────────────────────────────────┐
│  2. Product Search                      │
│     - Query by ID or barcode           │
│     - Load product details             │
└─────────────┬───────────────────────────┘
              │
              ▼
┌─────────────────────────────────────────┐
│  3. Store Quantity Lookup              │
│     - Get quantities across stores     │
│     - Filter positive quantities       │
│     - Display results                  │
└─────────────────────────────────────────┘
```

---

## 🌐 URL Routes & Actions

| URL Parameter | Function Called | Description |
|---------------|----------------|-------------|
| `do=` (empty) | Default | Search interface |
| `do=getDataAjax` | `getData()` | AJAX product lookup |

### AJAX Parameters
**Product Search** (`do=getDataAjax`):
- `search_text` - Barcode or product ID with prefix

---

## 🧮 Calculation Methods

### Barcode vs ID Detection
```php
// Arabic characters replacement
$text = str_replace("ه", "i", $text);
$text = str_replace("÷", "i", $text);
$text = str_replace("/", "i", $text);

// Prefix detection
if (in_array($text[0], ['i', 'I']) && $programSettings->usedParcode == 1) {
    $type = "id";
    $barcode = intval(trim($text, "i,I,ه"));
} else {
    $type = "barcode";
    $barcode = $text;
}
```

---

## 🔒 Security & Permissions

### Machine Interface
```php
$smarty->assign("itIsMachine_NoLogin", 1);
// No authentication required for machine access
```

### Input Sanitization
- Character replacement for Arabic input
- Integer casting for ID extraction
- Trim operations for clean processing

---

## 📊 Performance Considerations

### Optimizations
- Direct AJAX interface for quick lookups
- Minimal template loading
- Efficient product queries
- Store quantity filtering

### Considerations
- No pagination for store quantities
- Real-time inventory checking
- Minimal UI for machine interface

---

## 🐛 Common Issues & Troubleshooting

### 1. **Arabic Character Recognition**
**Issue**: Arabic characters not recognized in barcode scanners
**Solution**: Multiple character replacements implemented

### 2. **Store Quantity Filtering**
**Issue**: Empty stores showing in results
**Solution**: Filter with `productquantity > 0` condition

---

## 🧪 Testing Scenarios

### Test Case 1: Barcode Scanning
```
1. Scan product barcode
2. Verify product lookup
3. Check store quantities
4. Test with different barcode formats
```

### Test Case 2: ID Lookup
```
1. Enter ID with 'i' prefix
2. Test Arabic character input
3. Verify ID extraction
4. Check product details display
```

---

**Documented By**: AI Assistant  
**Review Status**: ✅ Complete  
**Next Review**: When major changes occur