# Product Review Controller Documentation

**File**: `/controllers/productreview.php`  
**Purpose**: Manages customer product reviews and ratings for online store integration  
**Last Updated**: December 20, 2024  
**Total Functions**: 3+  
**Lines of Code**: ~327

---

## 📋 Overview

The Product Review Controller manages customer product reviews and ratings. Features:
- Customer product review submission
- Rating system (1-5 stars)
- AJAX/cURL API support
- Review display and management
- Client-product relationship tracking

### Primary Functions
- [x] Add customer product reviews
- [x] Rating system implementation
- [x] AJAX/cURL API endpoints
- [x] Review history display
- [x] Client validation

---

## 🗄️ Database Tables

### Primary Tables
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **clientproductreview** | Customer reviews | id, clientid, productid, rating, review, sysDate |
| **client** | Customer master | clientid, clientname |
| **product** | Product master | productId, productName |

---

## 🔑 Key Functions

### 1. **add()** - Create Review
**Location**: Line 136  
**Purpose**: Add customer product review with validation

```php
function add() {
    $clientid = (int) $_POST['clientid'];
    $productid = (int) $_POST['productid'];
    $rating = (int) $_POST['rating'];
    $review = $_POST['review'];
    
    if ($clientid > 0 && $productid > 0) {
        $clientProductReview->clientid = $clientid;
        $clientProductReview->productid = $productid;
        $clientProductReview->rating = $rating;
        $clientProductReview->review = $review;
        $clientProductReview->sysDate = date("Y-m-d H:i:s");
        $id = $clientProductReviewDAO->insert($clientProductReview);
    }
    
    return $id;
}
```

### 2. **API Response Handling** - Multi-format Support
**Location**: Lines 94-107  
**Purpose**: Handle both web and API responses

```php
if (isset($_POST['curlpost']) && $_POST['curlpost'] == 1) {
    $data = array(
        'status' => 'success', 
        'status_code' => 200, 
        'message' => $httpStatusCodes[200], 
        'data' => ['id' => $id]
    );
    echo json_encode($data);
} else {
    header("location:?do=sucess");
}
```

---

## 🔄 Workflows

### Workflow: Review Submission
```
┌─────────────────────────────────────────┐
│       START: Submit Review             │
└─────────────┬───────────────────────────┘
              │
              ▼
┌─────────────────────────────────────────┐
│  1. Input Validation                    │
│     - Validate client ID               │
│     - Validate product ID              │
│     - Check rating range               │
└─────────────┬───────────────────────────┘
              │
              ▼
┌─────────────────────────────────────────┐
│  2. Review Creation                     │
│     - Insert review record             │
│     - Set system timestamp             │
│     - Return review ID                 │
└─────────────┬───────────────────────────┘
              │
              ▼
┌─────────────────────────────────────────┐
│  3. Response Handling                   │
│     - JSON for API calls               │
│     - Redirect for web interface       │
└─────────────────────────────────────────┘
```

---

## 🌐 URL Routes & Actions

| URL Parameter | Function Called | Description |
|---------------|----------------|-------------|
| `do=` (empty) or `do=show` | Default | Display interface |
| `do=add` | `add()` | Submit review |

### Required Parameters
**Add Review** (`do=add`):
- `clientid` - Customer ID (required, must be > 0)
- `productid` - Product ID (required, must be > 0)
- `rating` - Rating value (integer, typically 1-5)
- `review` - Review text (optional)

---

## 🧮 Calculation Methods

### Validation Logic
```php
if ($clientid > 0 && $productid > 0) {
    // Valid client and product IDs required
    // Proceed with review creation
}
```

### Rating System
- Integer rating value
- Typically 1-5 star system
- No range validation currently implemented

---

## 🔒 Security & Permissions

### Input Validation
```php
$clientid = (int) $_POST['clientid'];    // Integer casting
$productid = (int) $_POST['productid'];  // Integer casting
$rating = (int) $_POST['rating'];        // Integer casting
$review = $_POST['review'];              // String input
```

### API Security
- Supports cURL authentication
- JSON response format
- Proper HTTP status codes

---

## ⚠️ Issues Identified

### 1. **Incomplete Implementation**
- `showAllAjax()` function references undefined variables
- Template display logic incomplete
- Missing rating validation (no 1-5 range check)

### 2. **Security Considerations**
- No duplicate review prevention
- No review content sanitization
- Missing authorization checks

### 3. **Missing Features**
- No review editing capability
- No review moderation system
- No average rating calculation

---

## 🧪 Testing Scenarios

### Test Case 1: Basic Review Submission
```
1. Submit review with valid client/product IDs
2. Verify database insertion
3. Check API response format
4. Test web interface redirect
```

### Test Case 2: Validation Testing
```
1. Test with invalid client ID (0 or negative)
2. Test with invalid product ID
3. Test with missing parameters
4. Verify proper error handling
```

---

## 📚 Recommendations

### Immediate Improvements
1. **Add Rating Validation**:
   ```php
   if ($rating < 1 || $rating > 5) {
       throw new Exception("Rating must be between 1 and 5");
   }
   ```

2. **Prevent Duplicate Reviews**:
   ```sql
   CREATE UNIQUE INDEX idx_client_product_review 
   ON clientproductreview(clientid, productid);
   ```

3. **Add Content Sanitization**:
   ```php
   $review = htmlspecialchars($_POST['review'], ENT_QUOTES, 'UTF-8');
   ```

---

**Documented By**: AI Assistant  
**Review Status**: ⚠️ Needs Review - Incomplete implementation and security issues  
**Next Review**: Complete missing features and add security improvements