# Project Type Management Controller Documentation

**File**: `/controllers/projectKindController.php`  
**Purpose**: Manage project type definitions and categories  
**Last Updated**: 2024-12-20

---

## 📋 Overview

### Primary Functions
- [x] Project type creation
- [x] Project type listing
- [x] Project type editing
- [x] Project type soft deletion
- [x] YouTube link integration

### Related Controllers
- [projectController.php](projectController.md) - Uses project types
- [projectReportController.php](projectReportController.md) - Filters by type

---

## 🗄️ Database Tables

### Primary Tables
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| projectkind | Project type definitions | projectkindid, projectkindname, projectkindcomment, conditions |

### Secondary Tables (Referenced)
| Table Name | Purpose | Relationship |
|------------|---------|--------------|
| youtubelink | Help videos | General reference |

---

## 🔧 Key Functions

### 1. add()
**Purpose**: Create new project type  
**Called By**: POST action to `?do=add`  
**Parameters**:
- `$projectkindname` (string) - Type name
- `$projectkindcomment` (string) - Type description

**Returns**: Project kind ID

**Database Operations**:
- INSERT into `projectkind`

**Business Logic**:
1. Validate project type name
2. Set default conditions (active = 0)
3. Record creation timestamp and user
4. Insert new project type

**Example**:
```php
$_POST['projectkindname'] = 'Construction Project';
$_POST['projectkindcomment'] = 'Building and construction works';
add();
```

### 2. edit($id)
**Purpose**: Load project type for editing  
**Called By**: GET action to `?do=edit&id=$id`  
**Parameters**:
- `$id` (int) - Project kind ID

**Returns**: Project kind data

**Database Operations**:
- SELECT from `accountmovement` (Note: This appears to be an error)

**Business Logic**:
1. Load project kind data by ID
2. Return data for form population

### 3. tempdelete()
**Purpose**: Soft delete project type  
**Called By**: GET action to `?do=tempdelete&projectkindid=$id`  
**Parameters**:
- `$projectkindid` (int) - Project kind ID

**Returns**: Success/error redirect

**Database Operations**:
- UPDATE `projectkind` SET conditions = 1

**Business Logic**:
1. Mark project type as inactive
2. Preserve data for historical references

---

## 📊 Main Workflows

### Workflow 1: Project Type Creation
```
1. User accesses creation form
   ↓
2. Fill type name and description
   ↓
3. Submit form data
   ↓
4. Validate input fields
   ↓
5. Insert project type record
   ↓
6. Success/Error Response
```

**Files Involved**:
- View: `/views/default/projectkindview/add.html`
- Model: `/models/dto/Projectkind.class.php`
- DAO: `/models/mysql/ProjectkindMySqlDAO.class.php`

### Workflow 2: Project Type Management
```
1. Display project types list
   ↓
2. Select type for editing
   ↓
3. Load existing data
   ↓
4. Modify fields
   ↓
5. Save changes
   ↓
6. Update confirmation
```

---

## 🔗 File Dependencies

### Includes
```php
include("../public/impOpreation.php");
include_once("../public/config.php");
include_once("dailyentryfun.php");
include("../public/include_dao.php");
```

### Required Models
- Projectkind, YoutubeLink

### Related Views
- `/views/default/projectkindview/add.html` - Creation form
- `/views/default/projectkindview/show.html` - Types listing
- `/views/default/projectkindview/edit.html` - Edit form

---

## 🎯 URL Routes & Actions

| Action (`?do=`) | Method | Description | View Template |
|-----------------|--------|-------------|---------------|
| (empty) | GET | Display creation form | add.html |
| add | POST | Create new project type | - |
| show | GET | List all project types | show.html |
| tempdelete | GET | Soft delete project type | - |
| edit | GET | Display edit form | edit.html |
| sucess | GET | Success message | succes.html |
| error | GET | Error message | error.html |

---

## ⚠️ Known Issues & Fixes

### Issue 1: Wrong DAO Reference in Edit Function (Line 146)
**Problem**: Function references `$myAccountmovementRecord` instead of project kind DAO  
**Cause**: Copy-paste error from another controller  
**Fix**: Replace with `$myProjectkindRecord->load($id)`  
**File**: Lines 142-148

### Issue 2: Inconsistent Variable Naming
**Problem**: Mixed use of `$myProjectkind` and `$myProjectkindRecord`  
**Cause**: Inconsistent naming convention  
**Fix**: Standardize variable names  
**File**: Throughout controller

### Issue 3: Missing Rollback on Error (Line 77)
**Problem**: Transaction rollback called on undefined `$mytransactions`  
**Cause**: Transaction management not properly initialized  
**Fix**: Add proper transaction initialization or remove rollback  
**File**: Line 77

---

## 🔐 Permissions & Security

### Required Permissions
- Project type management access

### Security Checks
```php
include_once("../public/authentication.php");
```

### Input Validation
- All POST data accessed via `$_POST` (should use `filter_input()`)
- Session-based user tracking

---

## 📝 Notes

### Important Considerations
- Project types are used in project creation
- Soft deletion preserves historical data
- Arabic comments in code indicate client requirements
- YouTube links provide contextual help

### Future Improvements
- [ ] Add project type templates
- [ ] Implement type hierarchies
- [ ] Add color coding for types
- [ ] Enhance validation

### Database Schema
```sql
CREATE TABLE projectkind (
    projectkindid INT PRIMARY KEY AUTO_INCREMENT,
    projectkindname VARCHAR(255),
    projectkindcomment TEXT,
    conditions TINYINT DEFAULT 0,
    projectkinddate DATETIME,
    userid INT
);
```

---

## 📚 Related Documentation
- [Project Master](projectController.md)
- [Project Reports](projectReportController.md)
- [Authentication System](../authentication.md)