# Restaurant Hall Controller Documentation

**File**: `/controllers/restauranthall.php`  
**Purpose**: Manages restaurant halls/dining areas configuration and user access permissions  
**Last Updated**: December 19, 2024  
**Total Functions**: 6  
**Lines of Code**: 211

---

## 📋 Overview

The Restaurant Hall Controller manages dining hall configurations within the restaurant module. It handles:
- Creating and managing restaurant halls/dining areas
- Setting hall pricing configurations
- Managing user access permissions for each hall
- Integration with YouTube entertainment links for halls
- Soft deletion and hall lifecycle management

### Primary Functions
- [x] Create new restaurant halls
- [x] Edit existing hall configurations
- [x] Delete halls (soft delete)
- [x] View all halls with details
- [x] Manage hall-specific user permissions
- [x] Set hall pricing configurations
- [x] Integration with entertainment features

### Related Controllers
- [sellbillController.php](sellbillController.md) - Restaurant order processing
- [restauranttable.php](restauranttable.md) - Table management within halls
- [restaurantplaystation.php](restaurantplaystation.md) - Gaming station management
- [restaurantKitchenPrinter.php](restaurantKitchenPrinter.md) - Kitchen order printing
- [userController.php](userController.md) - User management for permissions
- [productController.php](productController.md) - Menu item management

---

## 🗄️ Database Tables

### Primary Tables (Direct Operations)
| Table Name | Purpose | Key Columns |
|------------|---------|-------------|
| **restauranthall** | Main hall configuration | id, name, hallPrice, hallUsers, userId, sysdate, del |

### Related Tables
| Table Name | Purpose | Relationship |
|------------|---------|--------------|
| **user** | System users | restauranthall.hallUsers (comma-separated IDs) |
| **youtubelink** | Entertainment links | Display integration for halls |
| **restauranttable** | Tables in halls | restauranttable.hallid → restauranthall.id |
| **restaurantplaystation** | Gaming stations | Associated with halls via tables |

### Table Structure Details
**restauranthall** columns:
- `id` - Primary key, auto-increment
- `name` - Hall name/identifier (VARCHAR)
- `hallPrice` - Base pricing for hall usage (DECIMAL)
- `hallUsers` - Comma-separated user IDs with access (TEXT)
- `userId` - Creator user ID (INT)
- `sysdate` - Creation date (DATE)
- `del` - Soft delete flag (TINYINT, 0=active, 1=deleted)

---

## 🔧 Key Functions

### 1. add()
**Purpose**: Create a new restaurant hall with user permissions  
**Called By**: Form submission with `?do=add`  
**Line**: 133-157

**Parameters** (via $_POST):
- `name` - Hall name
- `hallPrice` - Base hall pricing
- `hallUsers` - Array of user IDs with access

**Process Flow**:
```
┌─ Receive POST data ─┐
│                    │
├─ Extract hall name ─┤
│                    │
├─ Get hall price ───┤
│                    │
├─ Process user IDs ─┤
│  (array → string)  │
│                    │
├─ Set system fields ┤
│  (userId, date)    │
│                    │
└─ Insert to DB ─────┘
```

**Code Example**:
```php
$hallUsers = filter_input(INPUT_POST, 'hallUsers', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY);
$hallUsers_str = '-1,';
foreach ($hallUsers as $value) {
    $hallUsers_str .= $value . ",";
}
$hallUsers_str = rtrim($hallUsers_str, ',');

$restaurantHall->hallUsers = $hallUsers_str;
```

### 2. show()
**Purpose**: Retrieve all restaurant halls  
**Called By**: `?do=show`  
**Line**: 159-165

**Returns**: Array of all restaurant hall objects

**Process Flow**:
```
┌─ Query all halls ──┐
│                   │
├─ Get hall data ───┤
│                   │
└─ Return array ────┘
```

### 3. edit()
**Purpose**: Load hall data for editing  
**Called By**: `?do=edit&id=X`  
**Line**: 167-175

**Parameters**:
- `id` (GET) - Hall ID to edit

**Returns**: Hall object with exploded user array

**Process Flow**:
```
┌─ Get hall ID ─────┐
│                  │
├─ Load hall data ─┤
│                  │
├─ Explode users ──┤
│  (string → array)│
│                  │
└─ Return object ──┘
```

**Code Example**:
```php
$alldata = $restaurantHallDAO->load($id);
$alldata->hallUsers = explode(',', $alldata->hallUsers);
return $alldata;
```

### 4. update()
**Purpose**: Update existing hall configuration  
**Called By**: Form submission with `?do=update`  
**Line**: 177-203

**Parameters** (via $_POST):
- `id` - Hall ID to update
- `name` - Updated hall name
- `hallPrice` - Updated hall pricing
- `hallUsers` - Updated user permissions array

**Process Flow**:
```
┌─ Get hall ID ─────┐
│                  │
├─ Process form ───┤
│  data            │
│                  │
├─ Load existing ──┤
│  hall object     │
│                  │
├─ Update fields ──┤
│                  │
└─ Save to DB ─────┘
```

### 5. delete()
**Purpose**: Soft delete a restaurant hall  
**Called By**: `?do=delete&id=X`  
**Line**: 205-210

**Parameters**:
- `id` (GET) - Hall ID to delete

**Process Flow**:
```
┌─ Get hall ID ─────┐
│                  │
├─ Call DAO delete ┤
│                  │
└─ Remove record ──┘
```

### 6. Standard Controller Actions
**Purpose**: Handle routing and display  
**Lines**: 53-124

**Actions**:
- Default (empty do) - Show add form
- `add` - Process new hall creation  
- `show` - Display all halls with YouTube links
- `edit` - Show edit form
- `update` - Process hall updates
- `delete` - Remove hall
- `success/error` - Show result pages

---

## 🔄 Business Logic Flow

### Creating a New Hall
```
User Form Input
       │
       ▼
┌─────────────────┐    ┌──────────────────┐
│ Validate Input  │───▶│ Process Users    │
│ - Name required │    │ - Array to CSV   │
│ - Price numeric │    │ - Add -1 prefix  │
└─────────────────┘    └──────────────────┘
       │                        │
       ▼                        ▼
┌─────────────────┐    ┌──────────────────┐
│ Set Defaults    │───▶│ Insert to DB     │
│ - Current user  │    │ - Save hall      │
│ - Current date  │    │ - Get new ID     │
│ - del = 0       │    │                  │
└─────────────────┘    └──────────────────┘
```

### User Permission Management
```
User Selection (Checkboxes)
       │
       ▼
┌─────────────────┐
│ Filter Input    │ ← Uses FILTER_REQUIRE_ARRAY
│ - Get user IDs  │
│ - Validate      │
└─────────────────┘
       │
       ▼
┌─────────────────┐
│ Build String    │ ← Start with "-1," prefix
│ - Loop IDs      │
│ - Add commas    │
│ - Trim end      │
└─────────────────┘
       │
       ▼
┌─────────────────┐
│ Store in DB     │ ← hallUsers column
│ - CSV format    │
│ - Easy parsing  │
└─────────────────┘
```

---

## ⚠️ Common Issues

### 1. User Permission Format
**Issue**: User IDs must be stored as comma-separated string
**Solution**: Always use filter_input with FILTER_REQUIRE_ARRAY and convert to CSV format

### 2. Empty User Selection
**Issue**: No users selected causes empty array
**Solution**: Code includes -1 prefix to handle empty selections

### 3. Price Validation
**Issue**: Non-numeric prices can cause issues
**Solution**: Should add numeric validation on form input

---

## 🔗 Dependencies

### Required Files
- `../public/impOpreation.php` - Core operations
- `../public/config.php` - Configuration
- `../public/include_dao.php` - Database access objects
- `../public/authentication.php` - User authentication

### DAO Classes
- `RestauranthallMySqlDAO` - Database operations
- `RestauranthallMySqlExtDAO` - Extended database operations
- `UserMySqlDAO` - User data access
- `YoutubeLinkMySqlDAO` - Entertainment links

### Model Classes
- `Restauranthall.class.php` - Hall entity
- `User.class.php` - User entity
- `YoutubeLink.class.php` - Entertainment entity

### Templates
- `restauranthallview/add.html` - Add hall form
- `restauranthallview/show.html` - Hall listing
- `restauranthallview/edit.html` - Edit hall form
- `header.html` / `footer.html` - Layout templates
- `succes.html` / `error.html` - Result pages

---

**Integration Notes**:
- Halls serve as organizational units for restaurant tables
- User permissions control access to specific dining areas
- Price configuration supports different hall pricing models
- YouTube integration provides entertainment management
- Soft delete preserves data integrity for reporting