<?php
header('Content-Type: text/html; charset=utf-8');

include("../library/db_main/rb.php");
include("../models/sql/ConnectionProperty.class.php");

$host = ConnectionProperty::getHost();
$user = ConnectionProperty::getUser();
$password = ConnectionProperty::getPassword();
$database = ConnectionProperty::getDatabase();

R::setup("mysql:host=$host;dbname=$database;charset=latin1", $user, $password);
R::exec("SET NAMES latin1");

$controllerDir = realpath(__DIR__ . '/../controllers/') . '/';
$scannedFiles = @scandir($controllerDir);

if ($scannedFiles === false) {
    die("Error: Cannot read controllers directory: $controllerDir");
}

$controllersTemp = [];
foreach ($scannedFiles as $file) {
    if ($file === '.' || $file === '..') continue;
    $fullPath = $controllerDir . $file;
    if (is_file($fullPath) && pathinfo($file, PATHINFO_EXTENSION) === 'php') {
        $controllersTemp[] = $file;
    }
}
sort($controllersTemp);

$dependencies = [];
$tableDependencies = [];
$functionCalls = [];

foreach ($controllersTemp as $controller) {
    $fullPath = $controllerDir . $controller;
    if (!file_exists($fullPath)) continue;
    
    $content = file_get_contents($fullPath);
    $controllerName = basename($controller, '.php');
    
    preg_match_all('/require_once\s*\(\s*[\'"]\.\.\/models\/dao\/(\w+)DAO\.class\.php[\'"]\s*\)/', $content, $daoMatches);
    
    $tables = [];
    if (!empty($daoMatches[1])) {
        foreach ($daoMatches[1] as $table) {
            $tableName = strtolower($table);
            if (!isset($tableDependencies[$tableName])) {
                $tableDependencies[$tableName] = [];
            }
            $tableDependencies[$tableName][] = $controllerName;
            $tables[] = $table;
        }
    }
    
    preg_match_all('/function\s+(\w+)\s*\(/', $content, $funcMatches);
    
    $dependencies[$controllerName] = [
        'tables' => array_unique($tables),
        'functions' => $funcMatches[1] ?? []
    ];
}

$controllers = $controllersTemp;

$tables = R::getAll("SHOW TABLES");
$tableList = [];
foreach ($tables as $table) {
    $tableName = array_values($table)[0];
    $tableList[] = $tableName;
}

$selectedModule = isset($_GET['module']) ? $_GET['module'] : '';
$selectedTable = isset($_GET['table']) ? $_GET['table'] : '';
?>
<!DOCTYPE html>
<html dir="ltr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ERP Traceability Matrix & Impact Analysis</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            line-height: 1.6;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
        }
        
        .tabs {
            background: #2d3748;
            display: flex;
            padding: 0;
        }
        
        .tab {
            flex: 1;
            padding: 20px;
            text-align: center;
            color: white;
            cursor: pointer;
            transition: all 0.3s;
            border-right: 1px solid #4a5568;
        }
        
        .tab:last-child {
            border-right: none;
        }
        
        .tab:hover {
            background: #4a5568;
        }
        
        .tab.active {
            background: #667eea;
            font-weight: bold;
        }
        
        .content {
            padding: 40px;
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .search-box {
            background: #f7fafc;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 30px;
        }
        
        .search-box input, .search-box select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e2e8f0;
            border-radius: 6px;
            font-size: 16px;
            margin-bottom: 10px;
        }
        
        .search-box button {
            background: #667eea;
            color: white;
            padding: 12px 30px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 16px;
            transition: all 0.3s;
        }
        
        .search-box button:hover {
            background: #764ba2;
            transform: translateY(-2px);
        }
        
        .matrix-container {
            overflow-x: auto;
        }
        
        .matrix-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 14px;
        }
        
        .matrix-table th {
            background: #667eea;
            color: white;
            padding: 12px 8px;
            text-align: left;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        .matrix-table td {
            padding: 10px 8px;
            border: 1px solid #e2e8f0;
        }
        
        .matrix-table tr:nth-child(even) {
            background: #f7fafc;
        }
        
        .matrix-table tr:hover {
            background: #edf2f7;
        }
        
        .dependency-box {
            background: #fff;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .dependency-box h3 {
            color: #667eea;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #667eea;
        }
        
        .dependency-list {
            list-style: none;
            padding: 0;
        }
        
        .dependency-list li {
            padding: 10px;
            margin: 5px 0;
            background: #f7fafc;
            border-left: 4px solid #667eea;
            border-radius: 4px;
            transition: all 0.3s;
        }
        
        .dependency-list li:hover {
            background: #edf2f7;
            transform: translateX(5px);
        }
        
        .badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: bold;
            margin-left: 8px;
        }
        
        .badge.high {
            background: #feb2b2;
            color: #742a2a;
        }
        
        .badge.medium {
            background: #feebc8;
            color: #7c2d12;
        }
        
        .badge.low {
            background: #c6f6d5;
            color: #22543d;
        }
        
        .impact-level-high {
            background: #fed7d7 !important;
        }
        
        .impact-level-medium {
            background: #feebc8 !important;
        }
        
        .impact-level-low {
            background: #c6f6d5 !important;
        }
        
        .graph-container {
            background: white;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
            min-height: 400px;
        }
        
        .node {
            cursor: pointer;
        }
        
        .legend {
            display: flex;
            gap: 20px;
            margin-top: 20px;
            padding: 15px;
            background: #f7fafc;
            border-radius: 8px;
        }
        
        .legend-item {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .legend-color {
            width: 20px;
            height: 20px;
            border-radius: 4px;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 25px;
            border-radius: 12px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .stat-card h3 {
            font-size: 2.5em;
            margin-bottom: 5px;
        }
        
        .stat-card p {
            opacity: 0.9;
            font-size: 0.9em;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔍 ERP Traceability Matrix & Impact Analysis</h1>
            <p>Track dependencies and analyze the impact of changes across the system</p>
        </div>
        
        <div class="tabs">
            <div class="tab active" onclick="showTab('overview')">📊 Overview</div>
            <div class="tab" onclick="showTab('module-impact')">🎯 Module Impact</div>
            <div class="tab" onclick="showTab('table-usage')">🗄️ Table Usage</div>
            <div class="tab" onclick="showTab('dependency-graph')">🕸️ Dependency Graph</div>
        </div>
        
        <div class="content">
            <!-- Overview Tab -->
            <div id="overview" class="tab-content active">
                <h2 style="color: #667eea; margin-bottom: 20px;">System Overview</h2>
                
                <div class="stats-grid">
                    <div class="stat-card">
                        <h3><?php echo count($controllers); ?></h3>
                        <p>Total Controllers</p>
                    </div>
                    <div class="stat-card">
                        <h3><?php echo count($tableList); ?></h3>
                        <p>Database Tables</p>
                    </div>
                    <div class="stat-card">
                        <h3><?php echo count($tableDependencies); ?></h3>
                        <p>Tables in Use</p>
                    </div>
                    <div class="stat-card">
                        <h3>3</h3>
                        <p>Documented</p>
                    </div>
                </div>
                
                <div class="dependency-box">
                    <h3>📦 Most Used Tables</h3>
                    <div class="matrix-container">
                        <table class="matrix-table">
                            <thead>
                                <tr>
                                    <th>Table Name</th>
                                    <th>Used By Controllers</th>
                                    <th>Usage Count</th>
                                    <th>Impact Level</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $tableCounts = [];
                                foreach ($tableDependencies as $table => $ctrlList) {
                                    $tableCounts[$table] = count($ctrlList);
                                }
                                arsort($tableCounts);
                                $topTables = array_slice($tableCounts, 0, 20, true);
                                
                                foreach ($topTables as $table => $count) {
                                    $impactLevel = $count > 10 ? 'high' : ($count > 5 ? 'medium' : 'low');
                                    $impactClass = 'impact-level-' . $impactLevel;
                                    $usingControllers = $tableDependencies[$table];
                                    echo "<tr class='$impactClass'>";
                                    echo "<td><strong>" . htmlspecialchars($table) . "</strong></td>";
                                    echo "<td>" . implode(', ', array_slice($usingControllers, 0, 5)) . 
                                         (count($usingControllers) > 5 ? '...' : '') . "</td>";
                                    echo "<td><strong>$count</strong></td>";
                                    echo "<td><span class='badge $impactLevel'>" . strtoupper($impactLevel) . "</span></td>";
                                    echo "</tr>";
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
            <!-- Module Impact Tab -->
            <div id="module-impact" class="tab-content">
                <h2 style="color: #667eea; margin-bottom: 20px;">🎯 Module Impact Analysis</h2>
                
                <div class="search-box">
                    <label><strong>Select a Controller to Analyze:</strong></label>
                    
                    <?php
                    // DEBUG: Show controller count
                    echo "<p style='color: red; font-size: 12px;'>Debug: Found " . count($controllers) . " controllers. Type: " . gettype($controllers) . "</p>";
                    if (is_array($controllers)) {
                        echo "<p style='color: blue; font-size: 12px;'>First 5: " . implode(', ', array_slice($controllers, 0, 5)) . "</p>";
                    }
                    ?>
                    
                    <form method="GET" action="">
                        <input type="hidden" name="tab" value="module-impact">
                        <select name="module" onchange="this.form.submit()">
                            <option value="">-- Select Controller (<?php echo count($controllers); ?> found) --</option>
                            <?php foreach ($controllers as $ctrl): ?>
                                <option value="<?php echo htmlspecialchars($ctrl); ?>" 
                                    <?php echo $selectedModule === $ctrl ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($ctrl); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </form>
                </div>
                
                <?php 
                if ($selectedModule) {
                    $moduleName = basename($selectedModule, '.php');
                    
                    if (!isset($dependencies[$moduleName])) {
                        echo "<div class='dependency-box' style='background: #fff3cd; border-left: 4px solid #ffc107;'>";
                        echo "<p><strong>⚠️ No data found for this controller.</strong></p>";
                        echo "<p>Controller: " . htmlspecialchars($moduleName) . "</p>";
                        echo "<p>Available keys: " . implode(', ', array_keys($dependencies)) . "</p>";
                        echo "</div>";
                    }
                }
                
                if ($selectedModule && isset($dependencies[$moduleName])): 
                    $moduleData = $dependencies[$moduleName];
                ?>
                    <div class="dependency-box">
                        <h3>📋 Controller: <?php echo htmlspecialchars($moduleName); ?></h3>
                        <p><strong>Total Functions:</strong> <?php echo count($moduleData['functions']); ?></p>
                        <p><strong>Database Tables Used:</strong> <?php echo count($moduleData['tables']); ?></p>
                    </div>
                    
                    <div class="dependency-box">
                        <h3>🗄️ Database Tables Used</h3>
                        <ul class="dependency-list">
                            <?php foreach ($moduleData['tables'] as $table): 
                                $tableUsage = count($tableDependencies[strtolower($table)] ?? []);
                                $impactLevel = $tableUsage > 10 ? 'high' : ($tableUsage > 5 ? 'medium' : 'low');
                            ?>
                                <li>
                                    <strong><?php echo htmlspecialchars($table); ?></strong>
                                    <span class="badge <?php echo $impactLevel; ?>">
                                        Used by <?php echo $tableUsage; ?> controllers
                                    </span>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    
                    <div class="dependency-box">
                        <h3>⚠️ Impact Assessment</h3>
                        <p><strong>If you modify this controller:</strong></p>
                        <ul class="dependency-list">
                            <?php 
                            $affectedTables = $moduleData['tables'];
                            $affectedControllers = [];
                            foreach ($affectedTables as $table) {
                                $ctrlsUsingTable = $tableDependencies[strtolower($table)] ?? [];
                                foreach ($ctrlsUsingTable as $ctrl) {
                                    if ($ctrl !== $moduleName) {
                                        $affectedControllers[$ctrl] = true;
                                    }
                                }
                            }
                            ?>
                            <li class="impact-level-high">
                                <strong>Direct Impact:</strong> <?php echo count($affectedTables); ?> database tables
                            </li>
                            <li class="impact-level-medium">
                                <strong>Indirect Impact:</strong> <?php echo count($affectedControllers); ?> other controllers may be affected
                            </li>
                            <li class="impact-level-low">
                                <strong>Recommendation:</strong> Test all related modules after changes
                            </li>
                        </ul>
                        
                        <?php if (count($affectedControllers) > 0): ?>
                            <h4 style="margin-top: 20px; color: #e53e3e;">⚠️ Controllers That May Be Affected:</h4>
                            <ul class="dependency-list">
                                <?php foreach (array_keys($affectedControllers) as $ctrl): ?>
                                    <li><?php echo htmlspecialchars($ctrl); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Table Usage Tab -->
            <div id="table-usage" class="tab-content">
                <h2 style="color: #667eea; margin-bottom: 20px;">🗄️ Table Usage Analysis</h2>
                
                <div class="search-box">
                    <label><strong>Select a Database Table:</strong></label>
                    <form method="GET" action="">
                        <input type="hidden" name="tab" value="table-usage">
                        <select name="table" onchange="this.form.submit()">
                            <option value="">-- Select Table --</option>
                            <?php foreach ($tableList as $tbl): ?>
                                <option value="<?php echo htmlspecialchars($tbl); ?>"
                                    <?php echo $selectedTable === $tbl ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($tbl); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </form>
                </div>
                
                <?php if ($selectedTable): 
                    $tableControllers = $tableDependencies[strtolower($selectedTable)] ?? [];
                    $columns = R::getAll("DESCRIBE `$selectedTable`");
                ?>
                    <div class="dependency-box">
                        <h3>📊 Table: <?php echo htmlspecialchars($selectedTable); ?></h3>
                        <p><strong>Used by <?php echo count($tableControllers); ?> controllers</strong></p>
                        <p><strong>Columns:</strong> <?php echo count($columns); ?></p>
                    </div>
                    
                    <div class="dependency-box">
                        <h3>📋 Table Structure</h3>
                        <div class="matrix-container">
                            <table class="matrix-table">
                                <thead>
                                    <tr>
                                        <th>Field</th>
                                        <th>Type</th>
                                        <th>Null</th>
                                        <th>Key</th>
                                        <th>Default</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($columns as $col): ?>
                                        <tr>
                                            <td><strong><?php echo htmlspecialchars($col['Field']); ?></strong></td>
                                            <td><?php echo htmlspecialchars($col['Type']); ?></td>
                                            <td><?php echo htmlspecialchars($col['Null']); ?></td>
                                            <td><?php echo htmlspecialchars($col['Key']); ?></td>
                                            <td><?php echo htmlspecialchars($col['Default'] ?? 'NULL'); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    
                    <div class="dependency-box">
                        <h3>🎯 Controllers Using This Table</h3>
                        <ul class="dependency-list">
                            <?php foreach ($tableControllers as $ctrl): ?>
                                <li><?php echo htmlspecialchars($ctrl); ?>.php</li>
                            <?php endforeach; ?>
                        </ul>
                        
                        <?php if (count($tableControllers) > 0): ?>
                            <div style="margin-top: 20px; padding: 15px; background: #fff3cd; border-left: 4px solid #ffc107; border-radius: 4px;">
                                <strong>⚠️ Warning:</strong> Modifying this table structure will affect <strong><?php echo count($tableControllers); ?></strong> controller(s). 
                                Ensure thorough testing before deploying changes.
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Dependency Graph Tab -->
            <div id="dependency-graph" class="tab-content">
                <h2 style="color: #667eea; margin-bottom: 20px;">🕸️ Dependency Graph</h2>
                <p>Visual representation of system dependencies (Coming soon - will use D3.js or similar)</p>
                
                <div class="graph-container">
                    <p style="text-align: center; padding: 50px; color: #a0aec0;">
                        Interactive dependency graph visualization will be implemented here.<br>
                        It will show:
                        <ul style="list-style: none; margin-top: 20px;">
                            <li>✓ Controller → Table relationships</li>
                            <li>✓ Table → Controller dependencies</li>
                            <li>✓ Inter-controller dependencies</li>
                            <li>✓ Click nodes to explore connections</li>
                        </ul>
                    </p>
                </div>
                
                <div class="legend">
                    <div class="legend-item">
                        <div class="legend-color" style="background: #667eea;"></div>
                        <span>Controllers</span>
                    </div>
                    <div class="legend-item">
                        <div class="legend-color" style="background: #48bb78;"></div>
                        <span>Database Tables</span>
                    </div>
                    <div class="legend-item">
                        <div class="legend-color" style="background: #ed8936;"></div>
                        <span>High Impact</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        function showTab(tabName) {
            document.querySelectorAll('.tab').forEach(tab => tab.classList.remove('active'));
            document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));
            
            event.target.classList.add('active');
            document.getElementById(tabName).classList.add('active');
            
            const url = new URL(window.location);
            url.searchParams.set('tab', tabName);
            window.history.pushState({}, '', url);
        }
        
        window.addEventListener('load', function() {
            const urlParams = new URLSearchParams(window.location.search);
            const tab = urlParams.get('tab');
            if (tab) {
                const tabElement = document.querySelector(`[onclick="showTab('${tab}')"]`);
                if (tabElement) {
                    tabElement.click();
                }
            }
        });
    </script>
</body>
</html>
