<?php
header('Content-Type: text/html; charset=utf-8');

$controller = isset($_GET['controller']) ? $_GET['controller'] : 'buyBillController';
$mdFile = __DIR__ . "/controllers/{$controller}.md";

if (!file_exists($mdFile)) {
    die("Documentation file not found: {$controller}.md");
}

$markdown = file_get_contents($mdFile);

function parseFlowchart($text) {
    $lines = explode("\n", $text);
    $html = '<div class="flowchart-container">';
    
    foreach ($lines as $line) {
        $line = htmlspecialchars($line);
        
        if (strpos($line, '┌─') !== false || strpos($line, '└─') !== false) {
            $html .= '<div class="flow-box-border">' . $line . '</div>';
        } elseif (strpos($line, 'START:') !== false || strpos($line, 'END:') !== false) {
            $html .= '<div class="flow-step flow-start">' . strip_tags(str_replace(['│', '┌', '┐', '└', '┘', '─'], '', $line)) . '</div>';
        } elseif (preg_match('/^\s*│\s+(\d+)\.\s+(.+)│\s*$/', $line, $matches)) {
            $stepNum = $matches[1];
            $stepText = $matches[2];
            $html .= '<div class="flow-step" data-step="' . $stepNum . '"><span class="step-number">' . $stepNum . '</span>' . $stepText . '</div>';
        } elseif (preg_match('/│\s+(.+?)\s+│/', $line, $matches)) {
            $content = trim($matches[1]);
            if (strpos($content, 'FOR EACH') !== false || strpos($content, 'IF') !== false) {
                $html .= '<div class="flow-step flow-decision">' . $content . '</div>';
            } elseif (strpos($content, '├─→') !== false || strpos($content, '└─→') !== false) {
                $html .= '<div class="flow-substep">' . str_replace(['├─→', '└─→', '│'], ['→', '→', ''], $content) . '</div>';
            } else {
                $html .= '<div class="flow-step">' . $content . '</div>';
            }
        } elseif (trim($line) === '│' || strpos($line, '▼') !== false) {
            $html .= '<div class="flow-arrow">▼</div>';
        }
    }
    
    $html .= '</div>';
    return $html;
}

function parseMarkdown($text) {
    $placeholders = [];
    $counter = 0;
    
    $text = preg_replace_callback('/```(\w+)?\s*\n(.*?)\n```/s', function($matches) use (&$placeholders, &$counter) {
        $lang = isset($matches[1]) && $matches[1] ? $matches[1] : '';
        $code = $matches[2];
        
        if (strpos($code, '┌─') !== false || strpos($code, '│') !== false || strpos($code, 'START:') !== false) {
            $flowchart = parseFlowchart($code);
            $placeholder = "___FLOWCHART_{$counter}___";
            $placeholders[$placeholder] = $flowchart;
        } else {
            $code = htmlspecialchars($code);
            $displayLang = $lang ?: 'code';
            $placeholder = "___CODE_BLOCK_{$counter}___";
            $placeholders[$placeholder] = "<pre class='code-block {$displayLang}'><code>{$code}</code></pre>";
        }
        $counter++;
        return $placeholder;
    }, $text);
    
    $text = preg_replace_callback('/(\|.+\|[\r\n]+)((?:\|[-:\s|]+\|[\r\n]+))(((?:\|.+\|[\r\n]+)+))/m', function($matches) use (&$placeholders, &$counter) {
        $headerLine = trim($matches[1]);
        $bodyLines = trim($matches[3]);
        
        $headerCells = array_map('trim', explode('|', trim($headerLine, '|')));
        $bodyRows = array_filter(explode("\n", $bodyLines));
        
        $html = '<table><thead><tr>';
        foreach ($headerCells as $cell) {
            $html .= '<th>' . trim($cell) . '</th>';
        }
        $html .= '</tr></thead><tbody>';
        
        foreach ($bodyRows as $row) {
            $cells = array_map('trim', explode('|', trim($row, '|')));
            $html .= '<tr>';
            foreach ($cells as $cell) {
                $html .= '<td>' . trim($cell) . '</td>';
            }
            $html .= '</tr>';
        }
        
        $html .= '</tbody></table>';
        
        $placeholder = "___TABLE_{$counter}___";
        $placeholders[$placeholder] = $html;
        $counter++;
        return $placeholder;
    }, $text);
    
    $text = preg_replace('/`([^`\n]+)`/', '<code>$1</code>', $text);
    
    $text = preg_replace('/^# (.+)$/m', '<h1>$1</h1>', $text);
    $text = preg_replace('/^## (.+)$/m', '<h2>$1</h2>', $text);
    $text = preg_replace('/^### (.+)$/m', '<h3>$1</h3>', $text);
    $text = preg_replace('/^#### (.+)$/m', '<h4>$1</h4>', $text);
    
    $text = preg_replace('/\*\*(.+?)\*\*/', '<strong>$1</strong>', $text);
    $text = preg_replace('/(?<!\*)\*([^\*\n]+)\*(?!\*)/', '<em>$1</em>', $text);
    
    $text = preg_replace('/^\- \[x\] (.+)$/m', '<li class="checked">✅ $1</li>', $text);
    $text = preg_replace('/^\- \[ \] (.+)$/m', '<li class="unchecked">◻️ $1</li>', $text);
    $text = preg_replace('/^\- (.+)$/m', '<li>• $1</li>', $text);
    
    $text = preg_replace_callback('/\[(.+?)\]\((.+?)\)/', function($matches) {
        $url = $matches[2];
        $text = $matches[1];
        
        if (preg_match('/^(.+?)\.md$/', $url, $mdMatch)) {
            $controllerName = $mdMatch[1];
            $url = "?controller={$controllerName}";
        }
        
        return "<a href='{$url}'>{$text}</a>";
    }, $text);
    
    $text = preg_replace('/^---+$/m', '<hr>', $text);
    
    $lines = explode("\n", $text);
    $inList = false;
    $result = [];
    
    foreach ($lines as $line) {
        if (preg_match('/^<li/', $line)) {
            if (!$inList) {
                $result[] = '<ul>';
                $inList = true;
            }
            $result[] = $line;
        } else {
            if ($inList) {
                $result[] = '</ul>';
                $inList = false;
            }
            if (trim($line) !== '' && !preg_match('/^<(h[1-4]|hr|table|pre|div)/', $line)) {
                $line = '<p>' . $line . '</p>';
            }
            $result[] = $line;
        }
    }
    if ($inList) {
        $result[] = '</ul>';
    }
    
    $text = implode("\n", $result);
    
    foreach ($placeholders as $placeholder => $content) {
        $text = str_replace($placeholder, $content, $text);
    }
    
    return $text;
}

$html = parseMarkdown($markdown);
?>
<!DOCTYPE html>
<html dir="ltr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo ucfirst($controller); ?> Documentation</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            line-height: 1.6;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
        }
        
        .nav-container {
            background: #2d3748;
            padding: 20px 30px;
        }
        
        .search-box {
            margin-bottom: 20px;
        }
        
        .search-box input {
            width: 100%;
            padding: 12px 20px;
            font-size: 16px;
            border: 2px solid #4a5568;
            border-radius: 8px;
            background: #1a202c;
            color: white;
            outline: none;
            transition: all 0.3s;
        }
        
        .search-box input:focus {
            border-color: #667eea;
            box-shadow: 0 0 10px rgba(102, 126, 234, 0.5);
        }
        
        .search-box input::placeholder {
            color: #a0aec0;
        }
        
        .nav-categories {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        
        .category {
            background: #1a202c;
            border-radius: 8px;
            overflow: hidden;
        }
        
        .category-header {
            background: linear-gradient(135deg, #4a5568 0%, #2d3748 100%);
            color: white;
            padding: 12px 20px;
            cursor: pointer;
            font-weight: bold;
            font-size: 16px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s;
        }
        
        .category-header:hover {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        
        .category-header .toggle {
            transition: transform 0.3s;
        }
        
        .category.collapsed .toggle {
            transform: rotate(-90deg);
        }
        
        .category-content {
            display: flex;
            flex-direction: column;
            padding: 10px;
            gap: 5px;
            max-height: 500px;
            overflow-y: auto;
            transition: all 0.3s;
        }
        
        .category.collapsed .category-content {
            max-height: 0;
            padding: 0 10px;
            overflow: hidden;
        }
        
        .category-content a {
            color: white;
            text-decoration: none;
            padding: 10px 15px;
            background: #2d3748;
            border-radius: 5px;
            transition: all 0.3s;
            display: block;
        }
        
        .category-content a:hover {
            background: #667eea;
            transform: translateX(5px);
        }
        
        .category-content a.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            font-weight: bold;
        }
        
        .content {
            padding: 40px;
        }
        
        h1 {
            color: #667eea;
            font-size: 2.5em;
            margin: 30px 0 20px 0;
            border-bottom: 4px solid #667eea;
            padding-bottom: 10px;
        }
        
        h2 {
            color: #764ba2;
            font-size: 2em;
            margin: 30px 0 0 0;
            padding: 15px;
            padding-left: 50px;
            background: linear-gradient(135deg, #f7fafc 0%, #edf2f7 100%);
            border-left: 5px solid #764ba2;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s;
            position: relative;
        }
        
        h2:hover {
            background: linear-gradient(135deg, #edf2f7 0%, #e2e8f0 100%);
            transform: translateX(5px);
        }
        
        h2:before {
            content: '▼';
            position: absolute;
            left: 15px;
            transition: transform 0.3s;
        }
        
        h2.collapsed:before {
            transform: rotate(-90deg);
        }
        
        .section-content {
            max-height: 5000px;
            overflow: hidden;
            transition: max-height 0.5s ease-out, opacity 0.3s;
            opacity: 1;
            padding: 0 15px;
        }
        
        .section-content.collapsed {
            max-height: 0;
            opacity: 0;
            padding: 0;
        }
        
        h3 {
            color: #4a5568;
            font-size: 1.5em;
            margin: 25px 0 10px 0;
            padding-left: 10px;
            border-left: 3px solid #4a5568;
        }
        
        h4 {
            color: #718096;
            font-size: 1.2em;
            margin: 20px 0 10px 0;
        }
        
        p {
            margin: 10px 0;
            color: #2d3748;
            line-height: 1.8;
        }
        
        ul {
            margin: 15px 0;
            padding-left: 20px;
        }
        
        code {
            background: #f7fafc;
            padding: 2px 6px;
            border-radius: 3px;
            color: #e53e3e;
            font-family: 'Courier New', monospace;
            font-size: 0.9em;
        }
        
        .code-block {
            background: #2d3748;
            color: #68d391;
            padding: 20px;
            border-radius: 8px;
            overflow-x: auto;
            margin: 20px 0;
            border-left: 5px solid #667eea;
        }
        
        .code-block.php {
            border-left-color: #8993be;
        }
        
        .code-block.sql {
            border-left-color: #f6ad55;
        }
        
        .code-block code {
            background: transparent;
            color: #68d391;
            padding: 0;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        thead {
            background: #667eea;
            color: white;
        }
        
        th {
            padding: 15px;
            text-align: left;
            font-weight: bold;
            border-right: 1px solid rgba(255,255,255,0.2);
        }
        
        tbody tr {
            border-bottom: 1px solid #e2e8f0;
        }
        
        tbody tr:nth-child(even) {
            background: #f7fafc;
        }
        
        tbody tr:hover {
            background: #edf2f7;
        }
        
        td {
            padding: 12px;
            border-right: 1px solid #e2e8f0;
        }
        
        td:first-child {
            font-weight: bold;
            color: #667eea;
        }
        
        .flowchart-container {
            background: linear-gradient(135deg, #f7fafc 0%, #fff 100%);
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            padding: 30px;
            margin: 30px 0;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .flow-step {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px 25px;
            margin: 15px auto;
            border-radius: 12px;
            max-width: 600px;
            box-shadow: 0 4px 8px rgba(102, 126, 234, 0.3);
            position: relative;
            font-weight: 500;
            transition: all 0.3s;
        }
        
        .flow-step:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(102, 126, 234, 0.4);
        }
        
        .flow-step.flow-start {
            background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
            font-weight: bold;
            font-size: 1.1em;
            text-align: center;
        }
        
        .flow-step.flow-decision {
            background: linear-gradient(135deg, #f6ad55 0%, #ed8936 100%);
            border-radius: 8px;
            clip-path: polygon(10% 0%, 90% 0%, 100% 50%, 90% 100%, 10% 100%, 0% 50%);
            padding: 25px 40px;
        }
        
        .flow-substep {
            background: #edf2f7;
            color: #2d3748;
            padding: 12px 20px;
            margin: 8px auto 8px 60px;
            border-left: 4px solid #667eea;
            border-radius: 6px;
            max-width: 550px;
            font-size: 0.95em;
        }
        
        .flow-arrow {
            text-align: center;
            color: #667eea;
            font-size: 2em;
            margin: 5px 0;
            animation: bounce 2s infinite;
        }
        
        @keyframes bounce {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(5px); }
        }
        
        .step-number {
            background: rgba(255,255,255,0.3);
            padding: 5px 12px;
            border-radius: 20px;
            margin-right: 10px;
            font-weight: bold;
            font-size: 0.9em;
        }
        
        .flow-box-border {
            display: none;
        }
        
        li {
            margin: 8px 0;
            margin-left: 20px;
            color: #2d3748;
        }
        
        li.checked {
            color: #38a169;
            font-weight: bold;
            list-style: none;
            margin-left: 0;
        }
        
        li.unchecked {
            color: #a0aec0;
            list-style: none;
            margin-left: 0;
        }
        
        hr {
            border: none;
            border-top: 2px solid #e2e8f0;
            margin: 30px 0;
        }
        
        a {
            color: #667eea;
            text-decoration: none;
            border-bottom: 1px dotted #667eea;
            transition: all 0.3s;
        }
        
        a:hover {
            color: #764ba2;
            border-bottom-color: #764ba2;
        }
        
        strong {
            color: #2d3748;
            font-weight: 700;
        }
        
        .toc {
            background: #f7fafc;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
            border-left: 5px solid #667eea;
        }
        
        .toc h3 {
            color: #667eea;
            border: none;
            padding: 0;
            margin-top: 0;
        }
        
        .badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 0.85em;
            font-weight: bold;
            margin: 0 5px;
        }
        
        .badge.success {
            background: #c6f6d5;
            color: #22543d;
        }
        
        .badge.warning {
            background: #feebc8;
            color: #7c2d12;
        }
        
        .badge.info {
            background: #bee3f8;
            color: #2c5282;
        }
        
        .scroll-top {
            position: fixed;
            bottom: 30px;
            right: 30px;
            background: #667eea;
            color: white;
            width: 50px;
            height: 50px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
            transition: all 0.3s;
            opacity: 0;
            pointer-events: none;
        }
        
        .scroll-top.visible {
            opacity: 1;
            pointer-events: auto;
        }
        
        .scroll-top:hover {
            background: #764ba2;
            transform: translateY(-5px);
        }
        
        @media print {
            body {
                background: white;
            }
            .header, .nav, .scroll-top {
                display: none;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>📚 ERP Documentation Viewer</h1>
            <p>Beautiful, colorful documentation for your ERP system</p>
        </div>
        
        <div class="nav-container">
            <div class="search-box">
                <input type="text" id="docSearch" placeholder="🔍 Search controllers..." onkeyup="filterDocs()">
            </div>
            
            <div class="nav-categories">
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('inventory')">
                        📦 Inventory & Stock <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="inventory">
                        <a href="?controller=buyBillController" <?php echo $controller == 'buyBillController' ? 'class="active"' : ''; ?>>🛒 Buy Bills</a>
                        <a href="?controller=sellbillController" <?php echo $controller == 'sellbillController' ? 'class="active"' : ''; ?>>💰 Sell Bills</a>
                        <a href="?controller=returnBuyBillController" <?php echo $controller == 'returnBuyBillController' ? 'class="active"' : ''; ?>>↩️ Buy Returns</a>
                        <a href="?controller=returnSellBillController" <?php echo $controller == 'returnSellBillController' ? 'class="active"' : ''; ?>>🔙 Sell Returns</a>
                        <a href="?controller=storeController" <?php echo $controller == 'storeController' ? 'class="active"' : ''; ?>>🏬 Stores</a>
                        <a href="?controller=storedetailController" <?php echo $controller == 'storedetailController' ? 'class="active"' : ''; ?>>📊 Stock Details</a>
                        <a href="?controller=storemovementController" <?php echo $controller == 'storemovementController' ? 'class="active"' : ''; ?>>🔄 Stock Transfers</a>
                        <a href="?controller=catandproductstatistics" <?php echo $controller == 'catandproductstatistics' ? 'class="active"' : ''; ?>>📈 Data Quality Metrics</a>
                        <a href="?controller=companyproduces" <?php echo $controller == 'companyproduces' ? 'class="active"' : ''; ?>>🏭 Manufacturing Companies</a>
                        <a href="?controller=driveropticreport" <?php echo $controller == 'driveropticreport' ? 'class="active"' : ''; ?>>🚚 Workshop Delivery</a>
                        <a href="?controller=deleveryReportController" <?php echo $controller == 'deleveryReportController' ? 'class="active"' : ''; ?>>🔧 Maintenance Delivery</a>
                        <a href="?controller=MaintennanceSettlementController" <?php echo $controller == 'MaintennanceSettlementController' ? 'class="active"' : ''; ?>>💵 Maintenance Payments</a>
                        <a href="?controller=inventoryController" <?php echo $controller == 'inventoryController' ? 'class="active"' : ''; ?>>📊 Inventory Management</a>
                        <a href="?controller=inventorybybarcodeController" <?php echo $controller == 'inventorybybarcodeController' ? 'class="active"' : ''; ?>>🏷️ Barcode Inventory</a>
                        <a href="?controller=inventoryexpirationController" <?php echo $controller == 'inventoryexpirationController' ? 'class="active"' : ''; ?>>⏳ Expiration Management</a>
                        <a href="?controller=freebuys" <?php echo $controller == 'freebuys' ? 'class="active"' : ''; ?>>🎁 Free Purchases</a>
                        <a href="?controller=freesales" <?php echo $controller == 'freesales' ? 'class="active"' : ''; ?>>🎁 Free Sales</a>
                        <a href="?controller=fullCategoryReport" <?php echo $controller == 'fullCategoryReport' ? 'class="active"' : ''; ?>>📈 Category Analytics</a>
                        <a href="?controller=fixcats" <?php echo $controller == 'fixcats' ? 'class="active"' : ''; ?>>🔧 Category Utilities</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('product')">
                        🎁 Product Management <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="product">
                        <a href="?controller=productController" <?php echo $controller == 'productController' ? 'class="active"' : ''; ?>>📦 Products</a>
                        <a href="?controller=productCatController" <?php echo $controller == 'productCatController' ? 'class="active"' : ''; ?>>📂 Product Categories</a>
                        <a href="?controller=unitController" <?php echo $controller == 'unitController' ? 'class="active"' : ''; ?>>📏 Units</a>
                        <a href="?controller=productsParcodeController" <?php echo $controller == 'productsParcodeController' ? 'class="active"' : ''; ?>>🏷️ Barcodes</a>
                        <a href="?controller=productrequest" <?php echo $controller == 'productrequest' ? 'class="active"' : ''; ?>>📋 Product Requests</a>
                        <a href="?controller=productBuyreportController" <?php echo $controller == 'productBuyreportController' ? 'class="active"' : ''; ?>>📊 Purchase History</a>
                        <a href="?controller=productCatOnlineController" <?php echo $controller == 'productCatOnlineController' ? 'class="active"' : ''; ?>>🌐 Online Categories</a>
                        <a href="?controller=productImportFromExcelController" <?php echo $controller == 'productImportFromExcelController' ? 'class="active"' : ''; ?>>📥 Excel Import</a>
                        <a href="?controller=productParcodeOverImageFix" <?php echo $controller == 'productParcodeOverImageFix' ? 'class="active"' : ''; ?>>🔧 Barcode Print Fix</a>
                        <a href="?controller=productReduceImageSizeFix" <?php echo $controller == 'productReduceImageSizeFix' ? 'class="active"' : ''; ?>>🖼️ Image Optimization</a>
                        <a href="?controller=productcardController" <?php echo $controller == 'productcardController' ? 'class="active"' : ''; ?>>🎴 Product Cards</a>
                        <a href="?controller=productonline" <?php echo $controller == 'productonline' ? 'class="active"' : ''; ?>>🌐 Online Products</a>
                        <a href="?controller=productpriceaddreport" <?php echo $controller == 'productpriceaddreport' ? 'class="active"' : ''; ?>>💰 Price Addition</a>
                        <a href="?controller=productpricefix" <?php echo $controller == 'productpricefix' ? 'class="active"' : ''; ?>>🔧 Price Fix</a>
                        <a href="?controller=productsettingController" <?php echo $controller == 'productsettingController' ? 'class="active"' : ''; ?>>⚙️ Product Settings</a>
                        <a href="?controller=productunitupdate" <?php echo $controller == 'productunitupdate' ? 'class="active"' : ''; ?>>🔄 Unit Update</a>
                        <a href="?controller=productparcodefix" <?php echo $controller == 'productparcodefix' ? 'class="active"' : ''; ?>>🏷️ Barcode Fix</a>
                        <a href="?controller=productparcodefixusingid" <?php echo $controller == 'productparcodefixusingid' ? 'class="active"' : ''; ?>>🔢 ID-Based Barcode Fix</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('financial')">
                        💵 Financial Management <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="financial">
                        <a href="?controller=accountstree" <?php echo $controller == 'accountstree' ? 'class="active"' : ''; ?>>🌳 Chart of Accounts</a>
                        <a href="?controller=dailyentry" <?php echo $controller == 'dailyentry' ? 'class="active"' : ''; ?>>📒 Journal Entries</a>
                        <a href="?controller=expensesController" <?php echo $controller == 'expensesController' ? 'class="active"' : ''; ?>>💸 Expenses</a>
                        <a href="?controller=saveController" <?php echo $controller == 'saveController' ? 'class="active"' : ''; ?>>💰 Cash Management</a>
                        <a href="?controller=bankController" <?php echo $controller == 'bankController' ? 'class="active"' : ''; ?>>🏦 Banks</a>
                        <a href="?controller=bankaccountController" <?php echo $controller == 'bankaccountController' ? 'class="active"' : ''; ?>>🏦 Bank Accounts</a>
                        <a href="?controller=bankAccountDeficitController" <?php echo $controller == 'bankAccountDeficitController' ? 'class="active"' : ''; ?>>📉 Bank Deficit</a>
                        <a href="?controller=bankAccountDeficitKindController" <?php echo $controller == 'bankAccountDeficitKindController' ? 'class="active"' : ''; ?>>📋 Deficit Types</a>
                        <a href="?controller=bankAccountOpController" <?php echo $controller == 'bankAccountOpController' ? 'class="active"' : ''; ?>>📊 Bank Operations</a>
                        <a href="?controller=taxController" <?php echo $controller == 'taxController' ? 'class="active"' : ''; ?>>💳 Tax Management</a>
                        <a href="?controller=FaidaDireController" <?php echo $controller == 'FaidaDireController' ? 'class="active"' : ''; ?>>✅ Director Profit Approval</a>
                        <a href="?controller=dailyReportController" <?php echo $controller == 'dailyReportController' ? 'class="active"' : ''; ?>>📊 Daily Dashboard</a>
                        <a href="?controller=closingAccountController" <?php echo $controller == 'closingAccountController' ? 'class="active"' : ''; ?>>🔐 Account Closure</a>
                        <a href="?controller=dollarEarningsController" <?php echo $controller == 'dollarEarningsController' ? 'class="active"' : ''; ?>>💱 Currency Exchange</a>
                        <a href="?controller=costcenterController" <?php echo $controller == 'costcenterController' ? 'class="active"' : ''; ?>>🎯 Cost Centers</a>
                        <a href="?controller=costcenterReportController" <?php echo $controller == 'costcenterReportController' ? 'class="active"' : ''; ?>>📈 Cost Center Reports</a>
                        <a href="?controller=debtHistoryReportController" <?php echo $controller == 'debtHistoryReportController' ? 'class="active"' : ''; ?>>📅 Debt Aging</a>
                        <a href="?controller=disountreport" <?php echo $controller == 'disountreport' ? 'class="active"' : ''; ?>>💰 Discount Analysis</a>
                        <a href="?controller=expenseexchange" <?php echo $controller == 'expenseexchange' ? 'class="active"' : ''; ?>>💵 Project Expenses</a>
                        <a href="?controller=expensesTypeController" <?php echo $controller == 'expensesTypeController' ? 'class="active"' : ''; ?>>📂 Expense Types</a>
                        <a href="?controller=getSupplierDept" <?php echo $controller == 'getSupplierDept' ? 'class="active"' : ''; ?>>📊 Supplier Debt Utility</a>
                        <a href="?controller=incomeStatmentForPeriod" <?php echo $controller == 'incomeStatmentForPeriod' ? 'class="active"' : ''; ?>>📄 Income Statement</a>
                        <a href="?controller=incomeTypeController" <?php echo $controller == 'incomeTypeController' ? 'class="active"' : ''; ?>>📂 Income Types</a>
                        <a href="?controller=incomelist" <?php echo $controller == 'incomelist' ? 'class="active"' : ''; ?>>📊 Financial Dashboard</a>
                        <a href="?controller=generalledger" <?php echo $controller == 'generalledger' ? 'class="active"' : ''; ?>>📒 General Ledger</a>
                        <a href="?controller=invoicesearningsreportController" <?php echo $controller == 'invoicesearningsreportController' ? 'class="active"' : ''; ?>>💰 Invoice Earnings</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('contacts')">
                        👥 Contacts <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="contacts">
                        <a href="?controller=clientController" <?php echo $controller == 'clientController' ? 'class="active"' : ''; ?>>👥 Clients</a>
                        <a href="?controller=clientareaController" <?php echo $controller == 'clientareaController' ? 'class="active"' : ''; ?>>🗺️ Client Areas</a>
                        <a href="?controller=supplierController" <?php echo $controller == 'supplierController' ? 'class="active"' : ''; ?>>🏭 Suppliers</a>
                        <a href="?controller=clientpayments" <?php echo $controller == 'clientpayments' ? 'class="active"' : ''; ?>>💳 Client Payments</a>
                        <a href="?controller=clientreceiptController" <?php echo $controller == 'clientreceiptController' ? 'class="active"' : ''; ?>>🧾 Payment Receipts</a>
                        <a href="?controller=clientsAndsellProductsReportController" <?php echo $controller == 'clientsAndsellProductsReportController' ? 'class="active"' : ''; ?>>📊 Customer-Product Sales</a>
                        <a href="?controller=clientsWithoutBills" <?php echo $controller == 'clientsWithoutBills' ? 'class="active"' : ''; ?>>😴 Inactive Customers</a>
                        <a href="?controller=clientsexpensereport" <?php echo $controller == 'clientsexpensereport' ? 'class="active"' : ''; ?>>💸 Customer Expenses</a>
                        <a href="?controller=customerAccountMatching" <?php echo $controller == 'customerAccountMatching' ? 'class="active"' : ''; ?>>⚖️ Account Reconciliation</a>
                        <a href="?controller=customersproductsController" <?php echo $controller == 'customersproductsController' ? 'class="active"' : ''; ?>>📦 Customer-Product Analysis</a>
                        <a href="?controller=couponscontroller" <?php echo $controller == 'couponscontroller' ? 'class="active"' : ''; ?>>🎟️ Coupons</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('hr')">
                        👔 HR & Payroll <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="hr">
                        <a href="?controller=employeeController" <?php echo $controller == 'employeeController' ? 'class="active"' : ''; ?>>👥 Employees</a>
                        <a href="?controller=salaryReportController" <?php echo $controller == 'salaryReportController' ? 'class="active"' : ''; ?>>💰 Salary Processing</a>
                        <a href="?controller=EmployeeAttendanceController" <?php echo $controller == 'EmployeeAttendanceController' ? 'class="active"' : ''; ?>>⏰ Attendance</a>
                        <a href="?controller=employeePersonalController" <?php echo $controller == 'employeePersonalController' ? 'class="active"' : ''; ?>>💵 Loans & Advances</a>
                        <a href="?controller=employeeAttendance" <?php echo $controller == 'employeeAttendance' ? 'class="active"' : ''; ?>>📸 Real-time Attendance</a>
                        <a href="?controller=employeeAttendanceSystems" <?php echo $controller == 'employeeAttendanceSystems' ? 'class="active"' : ''; ?>>⚙️ Attendance Systems</a>
                        <a href="?controller=EmployeeAttendanceExcelController" <?php echo $controller == 'EmployeeAttendanceExcelController' ? 'class="active"' : ''; ?>>📊 Bulk Attendance Import</a>
                        <a href="?controller=employeeHolidays" <?php echo $controller == 'employeeHolidays' ? 'class="active"' : ''; ?>>🗓️ Holiday Calendar</a>
                        <a href="?controller=employeePersonalajex" <?php echo $controller == 'employeePersonalajex' ? 'class="active"' : ''; ?>>💼 Employee Financial AJAX</a>
                        <a href="?controller=employeedailyreport" <?php echo $controller == 'employeedailyreport' ? 'class="active"' : ''; ?>>📅 Daily Attendance Reports</a>
                        <a href="?controller=employeedoctorController" <?php echo $controller == 'employeedoctorController' ? 'class="active"' : ''; ?>>👨‍⚕️ Employee/Doctor Advances</a>
                        <a href="?controller=employeeendday" <?php echo $controller == 'employeeendday' ? 'class="active"' : ''; ?>>🌙 Shift Closure</a>
                        <a href="?controller=employeegroups" <?php echo $controller == 'employeegroups' ? 'class="active"' : ''; ?>>👥 Employee Groups</a>
                        <a href="?controller=employeereport" <?php echo $controller == 'employeereport' ? 'class="active"' : ''; ?>>📊 Employee Reports</a>
                        <a href="?controller=employeesubgroups" <?php echo $controller == 'employeesubgroups' ? 'class="active"' : ''; ?>>👤 Employee Subgroups</a>
                        <a href="?controller=executors" <?php echo $controller == 'executors' ? 'class="active"' : ''; ?>>⚙️ Sales Executors</a>
                        <a href="?controller=executorsreport" <?php echo $controller == 'executorsreport' ? 'class="active"' : ''; ?>>📈 Executor Reports</a>
                        <a href="?controller=executorsuserreport" <?php echo $controller == 'executorsuserreport' ? 'class="active"' : ''; ?>>👤 Executor User Analysis</a>
                        <a href="?controller=knownWaysController" <?php echo $controller == 'knownWaysController' ? 'class="active"' : ''; ?>>📍 Customer Acquisition</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('admin')">
                        ⚙️ System Administration <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="admin">
                        <a href="?controller=userController" <?php echo $controller == 'userController' ? 'class="active"' : ''; ?>>👤 Users</a>
                        <a href="?controller=usergroupController" <?php echo $controller == 'usergroupController' ? 'class="active"' : ''; ?>>👥 User Groups/Roles</a>
                        <a href="?controller=billsettingsController" <?php echo $controller == 'billsettingsController' ? 'class="active"' : ''; ?>>📄 Bill Templates</a>
                        <a href="?controller=programsettingsController" <?php echo $controller == 'programsettingsController' ? 'class="active"' : ''; ?>>⚙️ System Settings</a>
                        <a href="?controller=backupController" <?php echo $controller == 'backupController' ? 'class="active"' : ''; ?>>💾 Database Backup</a>
                        <a href="?controller=checkRedundantNames" <?php echo $controller == 'checkRedundantNames' ? 'class="active"' : ''; ?>>✅ Data Quality Check</a>
                        <a href="?controller=deactivate" <?php echo $controller == 'deactivate' ? 'class="active"' : ''; ?>>🔒 User Deactivation</a>
                        <a href="?controller=generalSearch" <?php echo $controller == 'generalSearch' ? 'class="active"' : ''; ?>>🔍 Universal Search</a>
                        <a href="?controller=exportexcel" <?php echo $controller == 'exportexcel' ? 'class="active"' : ''; ?>>📊 Excel Export</a>
                        <a href="?controller=fileController" <?php echo $controller == 'fileController' ? 'class="active"' : ''; ?>>📁 File Management</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('restaurant')">
                        🍽️ Restaurant/POS <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="restaurant">
                        <a href="?controller=restauranthall" <?php echo $controller == 'restauranthall' ? 'class="active"' : ''; ?>>🏛️ Halls</a>
                        <a href="?controller=restauranttable" <?php echo $controller == 'restauranttable' ? 'class="active"' : ''; ?>>🪑 Tables</a>
                        <a href="?controller=restaurantplaystation" <?php echo $controller == 'restaurantplaystation' ? 'class="active"' : ''; ?>>🎮 Gaming Stations</a>
                        <a href="?controller=restaurantKitchenPrinter" <?php echo $controller == 'restaurantKitchenPrinter' ? 'class="active"' : ''; ?>>🖨️ Kitchen Printer</a>
                        <a href="?controller=restoredController" <?php echo $controller == 'restoredController' ? 'class="active"' : ''; ?>>📋 Restaurant Orders</a>
                        <a href="?controller=cateResturantController" <?php echo $controller == 'cateResturantController' ? 'class="active"' : ''; ?>>📂 Restaurant Categories</a>
                        <a href="?controller=elemam" <?php echo $controller == 'elemam' ? 'class="active"' : ''; ?>>📊 Bill Reporting</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('manufacturing')">
                        🏭 Manufacturing/Production <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="manufacturing">
                        <a href="?controller=productionOrderController" <?php echo $controller == 'productionOrderController' ? 'class="active"' : ''; ?>>📋 Production Orders</a>
                        <a href="?controller=productionExecutionController" <?php echo $controller == 'productionExecutionController' ? 'class="active"' : ''; ?>>⚙️ Production Execution</a>
                        <a href="?controller=projectmaterialController" <?php echo $controller == 'projectmaterialController' ? 'class="active"' : ''; ?>>📦 Raw Materials</a>
                        <a href="?controller=simpleManufactureSettingsController" <?php echo $controller == 'simpleManufactureSettingsController' ? 'class="active"' : ''; ?>>🔧 Manufacturing Settings</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('reports')">
                        📊 Reports <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="reports">
                        <a href="?controller=FaidaController" <?php echo $controller == 'FaidaController' ? 'class="active"' : ''; ?>>💰 Profit Reports</a>
                        <a href="?controller=balancereportController" <?php echo $controller == 'balancereportController' ? 'class="active"' : ''; ?>>📈 Balance Sheet</a>
                        <a href="?controller=FinancialPositionController" <?php echo $controller == 'FinancialPositionController' ? 'class="active"' : ''; ?>>💼 Financial Position</a>
                        <a href="?controller=storereportController" <?php echo $controller == 'storereportController' ? 'class="active"' : ''; ?>>📦 Stock Reports</a>
                        <a href="?controller=clientReportsController" <?php echo $controller == 'clientReportsController' ? 'class="active"' : ''; ?>>👥 Client Account Reports</a>
                        <a href="?controller=clientReportsnewController" <?php echo $controller == 'clientReportsnewController' ? 'class="active"' : ''; ?>>👥 Enhanced Client Reports</a>
                        <a href="?controller=clientPayedDeptController" <?php echo $controller == 'clientPayedDeptController' ? 'class="active"' : ''; ?>>💳 Client Payment/Debt</a>
                        <a href="?controller=clientDeficitController" <?php echo $controller == 'clientDeficitController' ? 'class="active"' : ''; ?>>⚖️ Client Debt Settlement</a>
                        <a href="?controller=clientsWithProductsReport" <?php echo $controller == 'clientsWithProductsReport' ? 'class="active"' : ''; ?>>🔗 Client-Product Cross-Ref</a>
                        <a href="?controller=clientsAndProductsReportController" <?php echo $controller == 'clientsAndProductsReportController' ? 'class="active"' : ''; ?>>📊 Client-Product Analysis</a>
                        <a href="?controller=clientReportsbyareaController" <?php echo $controller == 'clientReportsbyareaController' ? 'class="active"' : ''; ?>>🗺️ Reports by Area</a>
                        <a href="?controller=supplierReportsController" <?php echo $controller == 'supplierReportsController' ? 'class="active"' : ''; ?>>🏭 Supplier Analytics</a>
                        <a href="?controller=supplierReportsnewController" <?php echo $controller == 'supplierReportsnewController' ? 'class="active"' : ''; ?>>🏭 Enhanced Supplier Reports</a>
                        <a href="?controller=productReportsController" <?php echo $controller == 'productReportsController' ? 'class="active"' : ''; ?>>📦 Product Analytics</a>
                        <a href="?controller=productserailreportController" <?php echo $controller == 'productserailreportController' ? 'class="active"' : ''; ?>>🔢 Product Serial Tracking</a>
                        <a href="?controller=productsellsreportController" <?php echo $controller == 'productsellsreportController' ? 'class="active"' : ''; ?>>📈 Product Sales Analysis</a>
                        <a href="?controller=productlimitreportController" <?php echo $controller == 'productlimitreportController' ? 'class="active"' : ''; ?>>⚠️ Stock Limit Alerts</a>
                        <a href="?controller=savedailyController" <?php echo $controller == 'savedailyController' ? 'class="active"' : ''; ?>>💰 Daily Cash Register</a>
                        <a href="?controller=cashflowsController" <?php echo $controller == 'cashflowsController' ? 'class="active"' : ''; ?>>💵 Cash Flow Reports</a>
                        <a href="?controller=expensesReportController" <?php echo $controller == 'expensesReportController' ? 'class="active"' : ''; ?>>💸 Expense Analytics</a>
                        <a href="?controller=storedetailpriceController" <?php echo $controller == 'storedetailpriceController' ? 'class="active"' : ''; ?>>💲 Store Pricing Details</a>
                        <a href="?controller=salesreportbills" <?php echo $controller == 'salesreportbills' ? 'class="active"' : ''; ?>>💵 Sales Bill Reports</a>
                        <a href="?controller=quickprofitreports" <?php echo $controller == 'quickprofitreports' ? 'class="active"' : ''; ?>>⚡ Quick Profit Analysis</a>
                        <a href="?controller=sellBillReportsController" <?php echo $controller == 'sellBillReportsController' ? 'class="active"' : ''; ?>>📊 Detailed Sales Reports</a>
                        <a href="?controller=buyBillreportController" <?php echo $controller == 'buyBillreportController' ? 'class="active"' : ''; ?>>🛒 Purchase Bill Reports</a>
                        <a href="?controller=userratereportController" <?php echo $controller == 'userratereportController' ? 'class="active"' : ''; ?>>👥 User Activity Reports</a>
                        <a href="?controller=ReturnSellBillReportsController" <?php echo $controller == 'ReturnSellBillReportsController' ? 'class="active"' : ''; ?>>🔙 Sales Return Reports</a>
                        <a href="?controller=absentReportController" <?php echo $controller == 'absentReportController' ? 'class="active"' : ''; ?>>📅 Absence Reports</a>
                        <a href="?controller=allReportsOnOne" <?php echo $controller == 'allReportsOnOne' ? 'class="active"' : ''; ?>>📊 Consolidated Reports</a>
                        <a href="?controller=allpremiumReportController" <?php echo $controller == 'allpremiumReportController' ? 'class="active"' : ''; ?>>💳 Premium Reports</a>
                        <a href="?controller=autoSalesReport" <?php echo $controller == 'autoSalesReport' ? 'class="active"' : ''; ?>>🤖 Auto Sales Report</a>
                        <a href="?controller=branchProfitabilityReport" <?php echo $controller == 'branchProfitabilityReport' ? 'class="active"' : ''; ?>>🏭 Branch Profitability</a>
                        <a href="?controller=buyBillReportsController" <?php echo $controller == 'buyBillReportsController' ? 'class="active"' : ''; ?>>📊 Purchase Reports Detail</a>
                        <a href="?controller=carPaymentsReportController" <?php echo $controller == 'carPaymentsReportController' ? 'class="active"' : ''; ?>>🚗 Vehicle Payments</a>
                        <a href="?controller=catsellrateReportController" <?php echo $controller == 'catsellrateReportController' ? 'class="active"' : ''; ?>>📈 Category Sales Rate</a>
                        <a href="?controller=checkReportController" <?php echo $controller == 'checkReportController' ? 'class="active"' : ''; ?>>📝 Check Reports</a>
                        <a href="?controller=clientPayedDeptReportController" <?php echo $controller == 'clientPayedDeptReportController' ? 'class="active"' : ''; ?>>💵 Client Payment Report</a>
                        <a href="?controller=knownwaysreportController" <?php echo $controller == 'knownwaysreportController' ? 'class="active"' : ''; ?>>📍 Acquisition Reports</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('other')">
                        🔧 Other <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="other">
                        <a href="?controller=incomeController" <?php echo $controller == 'incomeController' ? 'class="active"' : ''; ?>>💵 Income Management</a>
                        <a href="?controller=branchesController" <?php echo $controller == 'branchesController' ? 'class="active"' : ''; ?>>🏭 Branches</a>
                        <a href="?controller=currencyController" <?php echo $controller == 'currencyController' ? 'class="active"' : ''; ?>>💱 Currency</a>
                        <a href="?controller=typeClientController" <?php echo $controller == 'typeClientController' ? 'class="active"' : ''; ?>>🏷️ Client Types</a>
                        <a href="?controller=bills" <?php echo $controller == 'bills' ? 'class="active"' : ''; ?>>📄 Bills Management</a>
                        <a href="?controller=chartserp" <?php echo $controller == 'chartserp' ? 'class="active"' : ''; ?>>📊 Charts & Dashboards</a>
                        <a href="?controller=studentsReportdetails" <?php echo $controller == 'studentsReportdetails' ? 'class="active"' : ''; ?>>🎓 Student Reports</a>
                        <a href="?controller=bill" <?php echo $controller == 'bill' ? 'class="active"' : ''; ?>>📋 Manufacturing Bills</a>
                        <a href="?controller=buyBillfunction" <?php echo $controller == 'buyBillfunction' ? 'class="active"' : ''; ?>>🔧 Purchase Utilities</a>
                        <a href="?controller=buyBillupdate" <?php echo $controller == 'buyBillupdate' ? 'class="active"' : ''; ?>>🔄 Purchase Data Sync</a>
                        <a href="?controller=buyreport" <?php echo $controller == 'buyreport' ? 'class="active"' : ''; ?>>📊 Purchase Reports</a>
                        <a href="?controller=calcquickprofit" <?php echo $controller == 'calcquickprofit' ? 'class="active"' : ''; ?>>💰 Profit Calculator</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('assets')">
                        💼 Asset Management <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="assets">
                        <a href="?controller=assetController" <?php echo $controller == 'assetController' ? 'class="active"' : ''; ?>>💼 Fixed Assets</a>
                        <a href="?controller=assetTypeController" <?php echo $controller == 'assetTypeController' ? 'class="active"' : ''; ?>>📊 Asset Types</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('charity')">
                        🤝 Charity & Aid <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="charity">
                        <a href="?controller=beneficiariesController" <?php echo $controller == 'beneficiariesController' ? 'class="active"' : ''; ?>>👥 Beneficiaries</a>
                        <a href="?controller=aidTypeController" <?php echo $controller == 'aidTypeController' ? 'class="active"' : ''; ?>>🎁 Aid Types</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('maintenance')">
                        🔧 Maintenance & Service <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="maintenance">
                        <a href="?controller=maintenanceclients" <?php echo $controller == 'maintenanceclients' ? 'class="active"' : ''; ?>>👥 Maintenance Clients</a>
                        <a href="?controller=maintenancedeliverys" <?php echo $controller == 'maintenancedeliverys' ? 'class="active"' : ''; ?>>🚚 Delivery Operations</a>
                        <a href="?controller=maintenanceproducts" <?php echo $controller == 'maintenanceproducts' ? 'class="active"' : ''; ?>>📦 Parts Catalog</a>
                        <a href="?controller=maintenancereceipts" <?php echo $controller == 'maintenancereceipts' ? 'class="active"' : ''; ?>>📝 Service Receipts</a>
                        <a href="?controller=externalMaintenanceGoReport" <?php echo $controller == 'externalMaintenanceGoReport' ? 'class="active"' : ''; ?>>🏭 External Maintenance</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('insurance')">
                        🛡️ Insurance & Warranty <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="insurance">
                        <a href="?controller=insuranceGo" <?php echo $controller == 'insuranceGo' ? 'class="active"' : ''; ?>>📦 Warranty Outbound</a>
                        <a href="?controller=insuranceReturn" <?php echo $controller == 'insuranceReturn' ? 'class="active"' : ''; ?>>🔙 Warranty Returns</a>
                        <a href="?controller=insurancecompany" <?php echo $controller == 'insurancecompany' ? 'class="active"' : ''; ?>>🏭 Insurance Companies</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('einvoicing')">
                        🧾 Electronic Invoicing <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="einvoicing">
                        <a href="?controller=electronicinvoice" <?php echo $controller == 'electronicinvoice' ? 'class="active"' : ''; ?>>📱 E-Invoice Integration</a>
                        <a href="?controller=eBillRequests" <?php echo $controller == 'eBillRequests' ? 'class="active"' : ''; ?>>🔌 E-Bill API</a>
                        <a href="?controller=gt4setting" <?php echo $controller == 'gt4setting' ? 'class="active"' : ''; ?>>⚙️ GT4 Tax Settings</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('archive')">
                        🗄️ Archive & Documents <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="archive">
                        <a href="?controller=archive2Controller" <?php echo $controller == 'archive2Controller' ? 'class="active"' : ''; ?>>🗄️ Advanced Archive</a>
                        <a href="?controller=archiveController" <?php echo $controller == 'archiveController' ? 'class="active"' : ''; ?>>💾 Database Backup</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('accounting')">
                        📒 Accounting & Ledger <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="accounting">
                        <a href="?controller=assistantledger" <?php echo $controller == 'assistantledger' ? 'class="active"' : ''; ?>>📒 Account Ledger</a>
                        <a href="?controller=capitalController" <?php echo $controller == 'capitalController' ? 'class="active"' : ''; ?>>💰 Capital Management</a>
                        <a href="?controller=cashTransferController" <?php echo $controller == 'cashTransferController' ? 'class="active"' : ''; ?>>🔄 Cash Transfers</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('billmgmt')">
                        📄 Bill Management <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="billmgmt">
                        <a href="?controller=checkController" <?php echo $controller == 'checkController' ? 'class="active"' : ''; ?>>💵 Check Management</a>
                        <a href="?controller=billreceiptController" <?php echo $controller == 'billreceiptController' ? 'class="active"' : ''; ?>>📝 Maintenance Receipts</a>
                        <a href="?controller=billTemplateController" <?php echo $controller == 'billTemplateController' ? 'class="active"' : ''; ?>>📑 Bill Templates</a>
                        <a href="?controller=billoperationsControl" <?php echo $controller == 'billoperationsControl' ? 'class="active"' : ''; ?>>⚙️ Bill Operations</a>
                        <a href="?controller=billpropertyController" <?php echo $controller == 'billpropertyController' ? 'class="active"' : ''; ?>>🏷️ Bill Properties</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('purchase')">
                        🛒 Purchase Operations <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="purchase">
                        <a href="?controller=buyAndReturnBillController" <?php echo $controller == 'buyAndReturnBillController' ? 'class="active"' : ''; ?>>🔄 Buy & Return Bills</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('automotive')">
                        🚗 Vehicle Management <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="automotive">
                        <a href="?controller=carChasisController" <?php echo $controller == 'carChasisController' ? 'class="active"' : ''; ?>>🔢 Chassis Numbers</a>
                        <a href="?controller=carController" <?php echo $controller == 'carController' ? 'class="active"' : ''; ?>>🚗 Vehicle Inventory</a>
                        <a href="?controller=carReviewController" <?php echo $controller == 'carReviewController' ? 'class="active"' : ''; ?>>✅ Vehicle Review</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('reference')">
                        📊 Reference Data <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="reference">
                        <a href="?controller=governmentController" <?php echo $controller == 'governmentController' ? 'class="active"' : ''; ?>>🏛️ Governments</a>
                        <a href="?controller=associatedtag" <?php echo $controller == 'associatedtag' ? 'class="active"' : ''; ?>>🏷️ Tags</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('integration')">
                        🔗 System Integration <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="integration">
                        <a href="?controller=affectplugins" <?php echo $controller == 'affectplugins' ? 'class="active"' : ''; ?>>🔌 Financial Engine</a>
                        <a href="?controller=api" <?php echo $controller == 'api' ? 'class="active"' : ''; ?>>🔌 API Endpoints</a>
                        <a href="?controller=bind" <?php echo $controller == 'bind' ? 'class="active"' : ''; ?>>💼 Employee Custody</a>
                        <a href="?controller=boarding" <?php echo $controller == 'boarding' ? 'class="active"' : ''; ?>>🚀 Onboarding</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('pricing')">
                        🏷️ Pricing & Offers <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="pricing">
                        <a href="?controller=priceoffersController" <?php echo $controller == 'priceoffersController' ? 'class="active"' : ''; ?>>💰 Price Offers</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('delivery')">
                        🚚 Delivery & Shipping <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="delivery">
                        <a href="?controller=shippingFilesController" <?php echo $controller == 'shippingFilesController' ? 'class="active"' : ''; ?>>📦 Shipping Files</a>
                        <a href="?controller=shippercompaniesController" <?php echo $controller == 'shippercompaniesController' ? 'class="active"' : ''; ?>>🚚 Shipping Companies</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('orders')">
                        📎 Order Management <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="orders">
                        <a href="?controller=offerorder" <?php echo $controller == 'offerorder' ? 'class="active"' : ''; ?>>📦 Offer to Order</a>
                        <a href="?controller=offerclient" <?php echo $controller == 'offerclient' ? 'class="active"' : ''; ?>>📝 Client Offers</a>
                        <a href="?controller=orderedClientController" <?php echo $controller == 'orderedClientController' ? 'class="active"' : ''; ?>>👥 Client Orders</a>
                        <a href="?controller=orderClientsWithSelles" <?php echo $controller == 'orderClientsWithSelles' ? 'class="active"' : ''; ?>>📈 Client Sales Analysis</a>
                        <a href="?controller=orderSuppliersWithBuy" <?php echo $controller == 'orderSuppliersWithBuy' ? 'class="active"' : ''; ?>>📉 Supplier Purchase Analysis</a>
                        <a href="?controller=orderdeliveryreport" <?php echo $controller == 'orderdeliveryreport' ? 'class="active"' : ''; ?>>🚚 Delivery Reports</a>
                        <a href="?controller=orderperiodreport" <?php echo $controller == 'orderperiodreport' ? 'class="active"' : ''; ?>>📅 Reorder Period</a>
                        <a href="?controller=ordersreport" <?php echo $controller == 'ordersreport' ? 'class="active"' : ''; ?>>📊 Restaurant Orders</a>
                        <a href="?controller=orderstatusreport" <?php echo $controller == 'orderstatusreport' ? 'class="active"' : ''; ?>>📊 Order Status</a>
                        <a href="?controller=otherrequests" <?php echo $controller == 'otherrequests' ? 'class="active"' : ''; ?>>📝 Internal Requests</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('onlinestore')">
                        🛍️ Online Store <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="onlinestore">
                        <a href="?controller=onlineCatController" <?php echo $controller == 'onlineCatController' ? 'class="active"' : ''; ?>>📂 Online Catalog</a>
                        <a href="?controller=onlineDownloadsController" <?php echo $controller == 'onlineDownloadsController' ? 'class="active"' : ''; ?>>📊 Price Lists</a>
                        <a href="?controller=onlinestoremainsetting" <?php echo $controller == 'onlinestoremainsetting' ? 'class="active"' : ''; ?>>⚙️ Store Settings</a>
                        <a href="?controller=onlinestoresetting" <?php echo $controller == 'onlinestoresetting' ? 'class="active"' : ''; ?>>🔄 Sync Config</a>
                        <a href="?controller=onlinestoresync" <?php echo $controller == 'onlinestoresync' ? 'class="active"' : ''; ?>>🔌 Synchronization</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('maintenanceext')">
                        🔧 Maintenance Extended <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="maintenanceext">
                        <a href="?controller=maintenances" <?php echo $controller == 'maintenances' ? 'class="active"' : ''; ?>>📝 Maintenance Requests</a>
                        <a href="?controller=maintenancesends" <?php echo $controller == 'maintenancesends' ? 'class="active"' : ''; ?>>📦 Shipment Tracking</a>
                        <a href="?controller=maintenancesuppliers" <?php echo $controller == 'maintenancesuppliers' ? 'class="active"' : ''; ?>>🏭 Suppliers</a>
                        <a href="?controller=maintennanceArchiveController" <?php echo $controller == 'maintennanceArchiveController' ? 'class="active"' : ''; ?>>🗃️ Archive</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('transfer')">
                        🔄 Transfer & Movement <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="transfer">
                        <a href="?controller=movementmanageController" <?php echo $controller == 'movementmanageController' ? 'class="active"' : ''; ?>>📊 Activity Tracking</a>
                        <a href="?controller=netStoreTransfer" <?php echo $controller == 'netStoreTransfer' ? 'class="active"' : ''; ?>>🏭 Store Transfers</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('sysui')">
                        💻 System & UI <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="sysui">
                        <a href="?controller=menuurController" <?php echo $controller == 'menuurController' ? 'class="active"' : ''; ?>>📝 Menu Management</a>
                        <a href="?controller=noticesController" <?php echo $controller == 'noticesController' ? 'class="active"' : ''; ?>>🔔 Notifications</a>
                        <a href="?controller=noticefunctions" <?php echo $controller == 'noticefunctions' ? 'class="active"' : ''; ?>>🔔 Notice Functions</a>
                        <a href="?controller=pages" <?php echo $controller == 'pages' ? 'class="active"' : ''; ?>>📄 CMS Pages</a>
                        <a href="?controller=contactusController" <?php echo $controller == 'contactusController' ? 'class="active"' : ''; ?>>📧 Contact Forms</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('auth')">
                        🔐 Authentication <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="auth">
                        <a href="?controller=login" <?php echo $controller == 'login' ? 'class="active"' : ''; ?>>🔑 Login System</a>
                        <a href="?controller=loginfunction" <?php echo $controller == 'loginfunction' ? 'class="active"' : ''; ?>>🔧 Login Utilities</a>
                        <a href="?controller=logout" <?php echo $controller == 'logout' ? 'class="active"' : ''; ?>>🚪 Logout</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('dashboard')">
                        📊 Dashboard <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="dashboard">
                        <a href="?controller=index" <?php echo $controller == 'index' ? 'class="active"' : ''; ?>>🏠 Main Dashboard</a>
                        <a href="?controller=operationstotalreport" <?php echo $controller == 'operationstotalreport' ? 'class="active"' : ''; ?>>📊 Operations Total</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('finreports')">
                        📈 Financial Reports <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="finreports">
                        <a href="?controller=monthlytrialbalance" <?php echo $controller == 'monthlytrialbalance' ? 'class="active"' : ''; ?>>⚖️ Trial Balance</a>
                        <a href="?controller=mirrorReportOnCashAndBanks" <?php echo $controller == 'mirrorReportOnCashAndBanks' ? 'class="active"' : ''; ?>>💰 Cash Reconciliation</a>
                        <a href="?controller=marchSalaReport" <?php echo $controller == 'marchSalaReport' ? 'class="active"' : ''; ?>>💵 Salary Report</a>
                        <a href="?controller=networkreport" <?php echo $controller == 'networkreport' ? 'class="active"' : ''; ?>>💳 Payment Networks</a>
                        <a href="?controller=kempialaController" <?php echo $controller == 'kempialaController' ? 'class="active"' : ''; ?>>📝 Promissory Notes</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('sysconfig')">
                        ⚙️ System Config <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="sysconfig">
                        <a href="?controller=ex" <?php echo $controller == 'ex' ? 'class="active"' : ''; ?>>📧 Program Settings</a>
                        <a href="?controller=manualBackup" <?php echo $controller == 'manualBackup' ? 'class="active"' : ''; ?>>💾 Manual Backup</a>
                        <a href="?controller=manfacuresettingController" <?php echo $controller == 'manfacuresettingController' ? 'class="active"' : ''; ?>>🏭 Manufacturing Settings</a>
                        <a href="?controller=mechandiseReturnRequest" <?php echo $controller == 'mechandiseReturnRequest' ? 'class="active"' : ''; ?>>↩️ Return Requests</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('payment')">
                        💳 Payment & Premium <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="payment">
                        <a href="?controller=payedController" <?php echo $controller == 'payedController' ? 'class="active"' : ''; ?>>💳 Payments</a>
                        <a href="?controller=paymentMethodsController" <?php echo $controller == 'paymentMethodsController' ? 'class="active"' : ''; ?>>💵 Payment Methods</a>
                        <a href="?controller=premiumController" <?php echo $controller == 'premiumController' ? 'class="active"' : ''; ?>>📅 Premiums/Installments</a>
                        <a href="?controller=premiumReportController" <?php echo $controller == 'premiumReportController' ? 'class="active"' : ''; ?>>📊 Premium Reports</a>
                        <a href="?controller=premiumReportDelayController" <?php echo $controller == 'premiumReportDelayController' ? 'class="active"' : ''; ?>>⏰ Delayed Payments</a>
                        <a href="?controller=premiumclientController" <?php echo $controller == 'premiumclientController' ? 'class="active"' : ''; ?>>👥 Premium Clients</a>
                        <a href="?controller=premiumclientcheckboxController" <?php echo $controller == 'premiumclientcheckboxController' ? 'class="active"' : ''; ?>>☑️ Client Checkbox</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('partner')">
                        🤝 Partner Management <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="partner">
                        <a href="?controller=partnerController" <?php echo $controller == 'partnerController' ? 'class="active"' : ''; ?>>🤝 Partners</a>
                        <a href="?controller=partnerwithdrawalController" <?php echo $controller == 'partnerwithdrawalController' ? 'class="active"' : ''; ?>>💰 Withdrawals</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('production')">
                        🏭 Production <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="production">
                        <a href="?controller=productionOrderController" <?php echo $controller == 'productionOrderController' ? 'class="active"' : ''; ?>>📋 Production Orders</a>
                        <a href="?controller=productionExecutionController" <?php echo $controller == 'productionExecutionController' ? 'class="active"' : ''; ?>>⚙️ Execution</a>
                        <a href="?controller=projectmaterialController" <?php echo $controller == 'projectmaterialController' ? 'class="active"' : ''; ?>>📦 Materials</a>
                        <a href="?controller=simpleManufactureSettingsController" <?php echo $controller == 'simpleManufactureSettingsController' ? 'class="active"' : ''; ?>>⚙️ Settings</a>
                        <a href="?controller=productionEquationController" <?php echo $controller == 'productionEquationController' ? 'class="active"' : ''; ?>>🧮 Formulas</a>
                        <a href="?controller=productionOutController" <?php echo $controller == 'productionOutController' ? 'class="active"' : ''; ?>>📤 Output Tracking</a>
                        <a href="?controller=productionOutReportController" <?php echo $controller == 'productionOutReportController' ? 'class="active"' : ''; ?>>📊 Output Reports</a>
                        <a href="?controller=productionProcessController" <?php echo $controller == 'productionProcessController' ? 'class="active"' : ''; ?>>🔄 Processes</a>
                        <a href="?controller=productionReportController" <?php echo $controller == 'productionReportController' ? 'class="active"' : ''; ?>>📈 Production Analytics</a>
                        <a href="?controller=productionsettingController" <?php echo $controller == 'productionsettingController' ? 'class="active"' : ''; ?>>⚙️ Production Settings</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('entertainment')">
                        🎮 Entertainment <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="entertainment">
                        <a href="?controller=restaurantplaystation" <?php echo $controller == 'restaurantplaystation' ? 'class="active"' : ''; ?>>🎮 Gaming Stations</a>
                        <a href="?controller=playStationCardController" <?php echo $controller == 'playStationCardController' ? 'class="active"' : ''; ?>>🎫 Gaming Cards</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('datafix')">
                        🔧 Data Fix Utilities <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="datafix">
                        <a href="?controller=oldDataFix" <?php echo $controller == 'oldDataFix' ? 'class="active"' : ''; ?>>🔄 Legacy Data Migration</a>
                        <a href="?controller=oldTableToNewTable" <?php echo $controller == 'oldTableToNewTable' ? 'class="active"' : ''; ?>>📊 Schema Migration</a>
                        <a href="?controller=profitandlossCTRL" <?php echo $controller == 'profitandlossCTRL' ? 'class="active"' : ''; ?>>⚠️ Legacy P&L (Deprecated)</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('projects')">
                        📁 Project Management <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="projects">
                        <a href="?controller=projectController" <?php echo $controller == 'projectController' ? 'class="active"' : ''; ?>>📋 Projects</a>
                        <a href="?controller=projectKindController" <?php echo $controller == 'projectKindController' ? 'class="active"' : ''; ?>>🏷️ Project Types</a>
                        <a href="?controller=projectReportController" <?php echo $controller == 'projectReportController' ? 'class="active"' : ''; ?>>📊 Project Analytics</a>
                        <a href="?controller=projectclientsController" <?php echo $controller == 'projectclientsController' ? 'class="active"' : ''; ?>>👥 Project Clients</a>
                        <a href="?controller=projectdailyreportController" <?php echo $controller == 'projectdailyreportController' ? 'class="active"' : ''; ?>>📅 Daily Reports</a>
                        <a href="?controller=projectoperationController" <?php echo $controller == 'projectoperationController' ? 'class="active"' : ''; ?>>⚙️ Operations</a>
                        <a href="?controller=projectoperationControllerreturn" <?php echo $controller == 'projectoperationControllerreturn' ? 'class="active"' : ''; ?>>↩️ Returns</a>
                        <a href="?controller=projectstagesController" <?php echo $controller == 'projectstagesController' ? 'class="active"' : ''; ?>>📊 Stages</a>
                        <a href="?controller=projectstagesdataController" <?php echo $controller == 'projectstagesdataController' ? 'class="active"' : ''; ?>>📈 Stage Data</a>
                        <a href="?controller=projectreportControllerdetail" <?php echo $controller == 'projectreportControllerdetail' ? 'class="active"' : ''; ?>>📊 Detailed Reports</a>
                        <a href="?controller=projectreportControllerdetail2" <?php echo $controller == 'projectreportControllerdetail2' ? 'class="active"' : ''; ?>>📈 Extended Reports</a>
                        <a href="?controller=projectsaccountreport" <?php echo $controller == 'projectsaccountreport' ? 'class="active"' : ''; ?>>💰 Accounting</a>
                        <a href="?controller=projectApi" <?php echo $controller == 'projectApi' ? 'class="active"' : ''; ?>>🌐 Project API</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('profit')">
                        💹 Profit Analysis <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="profit">
                        <a href="?controller=profitdetailController" <?php echo $controller == 'profitdetailController' ? 'class="active"' : ''; ?>>💵 Profit Details</a>
                        <a href="?controller=profitproductController" <?php echo $controller == 'profitproductController' ? 'class="active"' : ''; ?>>📦 Product Profit</a>
                        <a href="?controller=profitproductcatController" <?php echo $controller == 'profitproductcatController' ? 'class="active"' : ''; ?>>📂 Category Profit</a>
                        <a href="?controller=profitproductcatControllernew" <?php echo $controller == 'profitproductcatControllernew' ? 'class="active"' : ''; ?>>🆕 Enhanced Category</a>
                        <a href="?controller=profitreportController" <?php echo $controller == 'profitreportController' ? 'class="active"' : ''; ?>>📊 Profit Reports</a>
                        <a href="?controller=profitandlossreport" <?php echo $controller == 'profitandlossreport' ? 'class="active"' : ''; ?>>📈 P&L Reports</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('bizops')">
                        🏢 Business Operations <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="bizops">
                        <a href="?controller=firms" <?php echo $controller == 'firms' ? 'class="active"' : ''; ?>>🏭 Firms</a>
                        <a href="?controller=propertiesController" <?php echo $controller == 'propertiesController' ? 'class="active"' : ''; ?>>🏗️ Properties</a>
                        <a href="?controller=propertyrightsreport" <?php echo $controller == 'propertyrightsreport' ? 'class="active"' : ''; ?>>📄 Property Rights</a>
                        <a href="?controller=ads" <?php echo $controller == 'ads' ? 'class="active"' : ''; ?>>📢 Advertisements</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('billutils')">
                        🧯 Bill Utilities <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="billutils">
                        <a href="?controller=billsfunctions" <?php echo $controller == 'billsfunctions' ? 'class="active"' : ''; ?>>🔧 Bill Functions</a>
                        <a href="?controller=billsfunctionsbuy" <?php echo $controller == 'billsfunctionsbuy' ? 'class="active"' : ''; ?>>🛍️ Buy Functions</a>
                        <a href="?controller=billsreturn" <?php echo $controller == 'billsreturn' ? 'class="active"' : ''; ?>>↩️ Return Operations</a>
                        <a href="?controller=bindsettlement" <?php echo $controller == 'bindsettlement' ? 'class="active"' : ''; ?>>💵 Cash Custody</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('checkdebt')">
                        💳 Check & Debt <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="checkdebt">
                        <a href="?controller=datedCheckedController" <?php echo $controller == 'datedCheckedController' ? 'class="active"' : ''; ?>>📅 Dated Checks</a>
                        <a href="?controller=depositcheckController" <?php echo $controller == 'depositcheckController' ? 'class="active"' : ''; ?>>🏦 Check Deposits</a>
                        <a href="?controller=debtclientController" <?php echo $controller == 'debtclientController' ? 'class="active"' : ''; ?>>📊 Client Debt</a>
                        <a href="?controller=clientdebtmessdiscoverer" <?php echo $controller == 'clientdebtmessdiscoverer' ? 'class="active"' : ''; ?>>🔍 Debt Discovery</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('stockcomment')">
                        📦 Stock & Comments <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="stockcomment">
                        <a href="?controller=otherstoremovementController" <?php echo $controller == 'otherstoremovementController' ? 'class="active"' : ''; ?>>🔄 Alt Stock Movement</a>
                        <a href="?controller=comentclientController" <?php echo $controller == 'comentclientController' ? 'class="active"' : ''; ?>>💬 Client Comments</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('acctutils')">
                        🧯 Accounting Utilities <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="acctutils">
                        <a href="?controller=dailyentryfun" <?php echo $controller == 'dailyentryfun' ? 'class="active"' : ''; ?>>🔧 Entry Functions</a>
                        <a href="?controller=dailyentrymany" <?php echo $controller == 'dailyentrymany' ? 'class="active"' : ''; ?>>📊 Bulk Entries</a>
                        <a href="?controller=reCalcBillPricesFunctions" <?php echo $controller == 'reCalcBillPricesFunctions' ? 'class="active"' : ''; ?>>🔄 Price Recalc</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('webapi')">
                        🌐 Web & API <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="webapi">
                        <a href="?controller=api_web" <?php echo $controller == 'api_web' ? 'class="active"' : ''; ?>>🌐 Web API</a>
                        <a href="?controller=initiateStaticSessionCommingWithCurl" <?php echo $controller == 'initiateStaticSessionCommingWithCurl' ? 'class="active"' : ''; ?>>🔄 CURL Sessions</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('healthcare')">
                        🏥 Healthcare <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="healthcare">
                        <a href="?controller=rb_kashf" <?php echo $controller == 'rb_kashf' ? 'class="active"' : ''; ?>>🩺 Medical Exams</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('ajaxops')">
                        ⚡ Ajax Operations <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="ajaxops">
                        <a href="?controller=expensesControllerajex" <?php echo $controller == 'expensesControllerajex' ? 'class="active"' : ''; ?>>💸 Expenses Ajax</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('realestate')">
                        🏢 Real Estate Management <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="realestate">
                        <a href="?controller=realestateunits" <?php echo $controller == 'realestateunits' ? 'class="active"' : ''; ?>>🏢 Property & Units</a>
                        <a href="?controller=realestateunitsowners" <?php echo $controller == 'realestateunitsowners' ? 'class="active"' : ''; ?>>💰 Property Purchases</a>
                        <a href="?controller=realestateunitsrenters" <?php echo $controller == 'realestateunitsrenters' ? 'class="active"' : ''; ?>>🏠 Property Rentals</a>
                        <a href="?controller=realestateunitsreports" <?php echo $controller == 'realestateunitsreports' ? 'class="active"' : ''; ?>>📊 Financial Reports</a>
                        <a href="?controller=realestateaveragerevenue" <?php echo $controller == 'realestateaveragerevenue' ? 'class="active"' : ''; ?>>💵 Revenue Analysis</a>
                        <a href="?controller=realestatepayments" <?php echo $controller == 'realestatepayments' ? 'class="active"' : ''; ?>>💳 Pending Payments</a>
                        <a href="?controller=realestateunitsalertsemptys" <?php echo $controller == 'realestateunitsalertsemptys' ? 'class="active"' : ''; ?>>🚨 Empty Properties</a>
                        <a href="?controller=realestateunitsalertsrents" <?php echo $controller == 'realestateunitsalertsrents' ? 'class="active"' : ''; ?>>⏰ Rental Alerts</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('rental')">
                        🎬 Equipment Rental <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="rental">
                        <a href="?controller=rentproducts" <?php echo $controller == 'rentproducts' ? 'class="active"' : ''; ?>>📦 Rental Inventory</a>
                        <a href="?controller=rentava" <?php echo $controller == 'rentava' ? 'class="active"' : ''; ?>>✅ Availability Tracking</a>
                        <a href="?controller=rentdelays" <?php echo $controller == 'rentdelays' ? 'class="active"' : ''; ?>>⏱️ Overdue Rentals</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('receipts')">
                        🧾 Receipt & Cash Management <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="receipts">
                        <a href="?controller=receiptController" <?php echo $controller == 'receiptController' ? 'class="active"' : ''; ?>>🧾 Receipt Processing</a>
                        <a href="?controller=saveCloseController" <?php echo $controller == 'saveCloseController' ? 'class="active"' : ''; ?>>💰 Register Closing</a>
                        <a href="?controller=saveReportController" <?php echo $controller == 'saveReportController' ? 'class="active"' : ''; ?>>📊 Cash Reports</a>
                        <a href="?controller=saveTypeController" <?php echo $controller == 'saveTypeController' ? 'class="active"' : ''; ?>>📂 Save Types</a>
                        <a href="?controller=saveTypeReportController" <?php echo $controller == 'saveTypeReportController' ? 'class="active"' : ''; ?>>📈 Type Reports</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('einvoicing')">
                        📄 E-Invoicing & Tax <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="einvoicing">
                        <a href="?controller=saudiElectronIcinvoiceController" <?php echo $controller == 'saudiElectronIcinvoiceController' ? 'class="active"' : ''; ?>>🇸🇦 ZATCA Integration</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('mfgextended')">
                        ⚙️ Manufacturing Extended <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="mfgextended">
                        <a href="?controller=settingOperationController" <?php echo $controller == 'settingOperationController' ? 'class="active"' : ''; ?>>⚙️ Operation Settings</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('searchsettlement')">
                        🔍 Search & Settlement <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="searchsettlement">
                        <a href="?controller=settlementKindSaveController" <?php echo $controller == 'settlementKindSaveController' ? 'class="active"' : ''; ?>>💼 Settlement Types</a>
                        <a href="?controller=searchFiltersController" <?php echo $controller == 'searchFiltersController' ? 'class="active"' : ''; ?>>🔍 Search Filters</a>
                    </div>
                </div>
                
                <div class="category collapsed">
                    <div class="category-header" onclick="toggleCategory('tools')">
                        🛠️ Tools <span class="toggle">▼</span>
                    </div>
                    <div class="category-content" id="tools">
                        <a href="../check_properties.php" target="_blank">🗂️ Properties Hierarchy</a>
                        <a href="../traceability_matrix.php" target="_blank">🔗 Traceability Matrix</a>
                        <a href="README.md" target="_blank">📖 Documentation Index</a>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="content">
            <?php echo $html; ?>
        </div>
    </div>
    
    <div class="scroll-top" id="scrollTop" onclick="scrollToTop()">
        ↑
    </div>
    
    <script>
        function toggleCategory(categoryId) {
            const category = document.getElementById(categoryId).parentElement;
            category.classList.toggle('collapsed');
        }
        
        function filterDocs() {
            const searchText = document.getElementById('docSearch').value.toLowerCase();
            const categories = document.querySelectorAll('.category');
            
            categories.forEach(category => {
                const links = category.querySelectorAll('.category-content a');
                let hasVisibleLink = false;
                
                links.forEach(link => {
                    const text = link.textContent.toLowerCase();
                    if (text.includes(searchText)) {
                        link.style.display = 'block';
                        hasVisibleLink = true;
                    } else {
                        link.style.display = 'none';
                    }
                });
                
                if (searchText === '') {
                    category.style.display = 'block';
                    category.classList.remove('collapsed');
                } else if (hasVisibleLink) {
                    category.style.display = 'block';
                    category.classList.remove('collapsed');
                } else {
                    category.style.display = 'none';
                }
            });
        }
        
        window.addEventListener('scroll', function() {
            const scrollTop = document.getElementById('scrollTop');
            if (window.pageYOffset > 300) {
                scrollTop.classList.add('visible');
            } else {
                scrollTop.classList.remove('visible');
            }
        });
        
        function scrollToTop() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        }
        
        
        function makeAccordions() {
            const contentDiv = document.querySelector('.content');
            const h2Elements = Array.from(contentDiv.querySelectorAll('h2'));
            
            h2Elements.forEach(function(h2) {
                const wrapper = document.createElement('div');
                wrapper.className = 'section-content collapsed';
                
                let nextElement = h2.nextElementSibling;
                
                while (nextElement && nextElement.tagName !== 'H2' && nextElement.tagName !== 'H1') {
                    const currentElement = nextElement;
                    nextElement = nextElement.nextElementSibling;
                    wrapper.appendChild(currentElement);
                }
                
                h2.parentNode.insertBefore(wrapper, h2.nextSibling);
                
                h2.classList.add('collapsed');
                
                h2.addEventListener('click', function() {
                    this.classList.toggle('collapsed');
                    wrapper.classList.toggle('collapsed');
                });
            });
        }
        
        window.addEventListener('load', function() {
            setTimeout(makeAccordions, 200);
        });
        
        const expandAllBtn = document.createElement('button');
        expandAllBtn.innerHTML = '📂 Expand All';
        expandAllBtn.style.cssText = 'position: fixed; bottom: 90px; right: 30px; background: #48bb78; color: white; padding: 12px 20px; border: none; border-radius: 25px; cursor: pointer; box-shadow: 0 4px 12px rgba(0,0,0,0.3); font-weight: bold; z-index: 1000; transition: all 0.3s;';
        document.body.appendChild(expandAllBtn);
        
        const collapseAllBtn = document.createElement('button');
        collapseAllBtn.innerHTML = '📁 Collapse All';
        collapseAllBtn.style.cssText = 'position: fixed; bottom: 150px; right: 30px; background: #f56565; color: white; padding: 12px 20px; border: none; border-radius: 25px; cursor: pointer; box-shadow: 0 4px 12px rgba(0,0,0,0.3); font-weight: bold; z-index: 1000; transition: all 0.3s;';
        document.body.appendChild(collapseAllBtn);
        
        expandAllBtn.addEventListener('click', function() {
            document.querySelectorAll('.content h2').forEach(function(h2) {
                h2.classList.remove('collapsed');
                const nextDiv = h2.nextElementSibling;
                if (nextDiv && nextDiv.classList.contains('section-content')) {
                    nextDiv.classList.remove('collapsed');
                }
            });
        });
        
        collapseAllBtn.addEventListener('click', function() {
            document.querySelectorAll('.content h2').forEach(function(h2) {
                h2.classList.add('collapsed');
                const nextDiv = h2.nextElementSibling;
                if (nextDiv && nextDiv.classList.contains('section-content')) {
                    nextDiv.classList.add('collapsed');
                }
            });
        });
        
        expandAllBtn.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-3px)';
            this.style.background = '#38a169';
        });
        expandAllBtn.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
            this.style.background = '#48bb78';
        });
        
        collapseAllBtn.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-3px)';
            this.style.background = '#e53e3e';
        });
        collapseAllBtn.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
            this.style.background = '#f56565';
        });
    </script>
</body>
</html>
