/**
 * Sidebar Menu with Keyboard Shortcut
 * Hotkey: Alt + M to toggle sidebar
 */

(function() {
    'use strict';
    
    // Wait for DOM to be ready
    document.addEventListener('DOMContentLoaded', initSidebar);
    
    function initSidebar() {
        const sidebar = document.querySelector('.sidebar-menu');
        const mainContent = document.querySelector('.main-content');
        const toggleBtn = document.querySelector('.sidebar-toggle');
        const overlay = document.querySelector('.sidebar-overlay');
        
        if (!sidebar) {
            console.error('Sidebar menu not found');
            return;
        }
        
        console.log('✓ Sidebar initialized');
        
        // Load saved state from localStorage
        const savedState = localStorage.getItem('sidebarCollapsed');
        if (savedState === 'true') {
            sidebar.classList.add('collapsed');
            if (mainContent) {
                mainContent.classList.add('sidebar-collapsed');
            }
            console.log('✓ Loaded collapsed state from localStorage');
        }
        
        // Toggle function
        window.toggleSidebar = function() {
            const isCollapsed = sidebar.classList.contains('collapsed');
            console.log('🔄 Toggle sidebar - currently collapsed:', isCollapsed);
            
            if (isCollapsed) {
                // EXPAND
                sidebar.classList.remove('collapsed');
                if (mainContent) {
                    mainContent.classList.remove('sidebar-collapsed');
                }
                localStorage.setItem('sidebarCollapsed', 'false');
                console.log('✓ Sidebar EXPANDED');
            } else {
                // COLLAPSE
                sidebar.classList.add('collapsed');
                if (mainContent) {
                    mainContent.classList.add('sidebar-collapsed');
                }
                localStorage.setItem('sidebarCollapsed', 'true');
                
                // Close all open submenus
                document.querySelectorAll('.sidebar-menu .dropdown.open').forEach(li => {
                    li.classList.remove('open');
                });
                console.log('✓ Sidebar COLLAPSED');
            }
        };
        
        // Toggle button click
        if (toggleBtn) {
            toggleBtn.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                console.log('🖱️ Toggle button clicked');
                window.toggleSidebar();
            });
            console.log('✓ Toggle button listener attached');
        } else {
            console.warn('⚠️ Toggle button not found');
        }
        
        // Keyboard shortcut: Alt + M
        document.addEventListener('keydown', function(e) {
            if (e.altKey && e.key.toLowerCase() === 'm') {
                e.preventDefault();
                console.log('⌨️ Alt+M pressed');
                window.toggleSidebar();
            }
        });
        console.log('✓ Keyboard shortcut (Alt+M) registered');
        
        // Menu item click - toggle submenu (main dropdown menus)
        const menuItems = document.querySelectorAll('.sidebar-menu .nav > li.dropdown > a');
        menuItems.forEach(item => {
            item.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                const parent = this.parentElement;
                const wasOpen = parent.classList.contains('open');
                
                console.log('📂 Main menu clicked:', this.textContent.trim().substring(0, 30), 'wasOpen:', wasOpen);
                
                // Close ALL other open menus at this level
                document.querySelectorAll('.sidebar-menu .nav > li.dropdown.open').forEach(li => {
                    li.classList.remove('open');
                });
                
                // Toggle current menu (open if was closed)
                if (!wasOpen) {
                    parent.classList.add('open');
                    console.log('✓ Opened main menu');
                } else {
                    console.log('✓ Closed main menu');
                }
                
                // If sidebar is collapsed, expand it
                if (sidebar.classList.contains('collapsed')) {
                    console.log('Expanding collapsed sidebar...');
                    window.toggleSidebar();
                }
            });
        });
        console.log('✓ Main menu listeners attached (' + menuItems.length + ' items)');
        
        // Add title attribute for tooltips when collapsed
        menuItems.forEach(item => {
            const text = item.textContent.trim();
            if (text) {
                const cleanText = text.replace(/▼/g, '').replace(/\s+/g, ' ').trim();
                item.setAttribute('title', cleanText);
            }
        });
        console.log('✓ Tooltips added for collapsed mode');
        
        // Search functionality - search only parent nodes (exclude leaf/final items)
        const searchInput = document.getElementById('sidebar-search-input');
        if (searchInput) {
            searchInput.addEventListener('input', function() {
                const searchTerm = this.value.toLowerCase().trim();
                
                if (searchTerm === '') {
                    // Show all items
                    const allItems = sidebar.querySelectorAll('.nav li, .dropdown-menu li, .sub-menu li');
                    allItems.forEach(item => {
                        item.classList.remove('hidden-by-search');
                    });
                } else {
                    // Get ALL menu items
                    const allItems = sidebar.querySelectorAll('.sidebar-menu li');
                    
                    // Hide all items first
                    allItems.forEach(item => {
                        item.classList.add('hidden-by-search');
                    });
                    
                    // Find items that have submenus (parent nodes only)
                    allItems.forEach(item => {
                        const hasSubmenu = item.querySelector('.sub-menu, .dropdown-menu');
                        
                        if (hasSubmenu) {
                            // This is a parent node - search it
                            const link = item.querySelector(':scope > a');
                            if (link) {
                                const text = link.textContent.toLowerCase();
                                if (text.includes(searchTerm)) {
                                    // Show this parent and ALL its children
                                    item.classList.remove('hidden-by-search');
                                    item.classList.add('open');
                                    
                                    // Show all descendants
                                    const allChildren = item.querySelectorAll('li');
                                    allChildren.forEach(child => {
                                        child.classList.remove('hidden-by-search');
                                    });
                                    
                                    // Show all parent items up the tree
                                    let parent = item.parentElement;
                                    while (parent && !parent.classList.contains('sidebar-menu')) {
                                        if (parent.tagName === 'LI') {
                                            parent.classList.remove('hidden-by-search');
                                            parent.classList.add('open');
                                        }
                                        parent = parent.parentElement;
                                    }
                                }
                            }
                        }
                    });
                }
            });
            console.log('✓ Search functionality added');
        }
        
        // Handle ALL nested submenu clicks (any level) - CLICK ONLY, NO HOVER
        // This handles 2nd, 3rd, 4th level menus
        sidebar.addEventListener('click', function(e) {
            // Find if we clicked on a link
            const clickedLink = e.target.closest('a');
            
            if (clickedLink && clickedLink.closest('.sidebar-menu')) {
                const parent = clickedLink.parentElement;
                
                // Check if this link has a submenu (has .sub-menu or .dropdown-menu child)
                const hasSubmenu = parent.querySelector('.sub-menu, .dropdown-menu');
                
                if (hasSubmenu) {
                    // This is a dropdown toggle - prevent navigation and toggle submenu
                    e.preventDefault();
                    e.stopPropagation();
                    
                    const wasOpen = parent.classList.contains('open');
                    console.log('📁 Submenu toggle clicked:', clickedLink.textContent.trim().substring(0, 30), 'wasOpen:', wasOpen);
                    
                    // Close other open submenus at same level (siblings only)
                    const parentUl = parent.parentElement;
                    if (parentUl) {
                        const siblings = parentUl.querySelectorAll(':scope > li.open');
                        siblings.forEach(li => {
                            if (li !== parent) {
                                li.classList.remove('open');
                                console.log('Closed sibling menu');
                            }
                        });
                    }
                    
                    // Toggle current submenu
                    if (!wasOpen) {
                        parent.classList.add('open');
                        console.log('✓ Opened submenu');
                    } else {
                        parent.classList.remove('open');
                        console.log('✓ Closed submenu');
                    }
                    
                    return false;
                } else {
                    // This is a regular link - let it navigate
                    console.log('🔗 Regular link clicked, navigating to:', clickedLink.getAttribute('href'));
                    // Don't prevent default - allow navigation
                }
            }
        });
        
        const submenuCount = document.querySelectorAll('.sidebar-menu .dropdown-menu a.dropdown-toggle').length;
        console.log('✓ Nested menu listeners attached (' + submenuCount + ' items)');
        
        // Overlay click (mobile)
        if (overlay) {
            overlay.addEventListener('click', function() {
                sidebar.classList.remove('show');
                overlay.classList.remove('show');
            });
        }
        
        // Mobile menu toggle
        const mobileMenuBtn = document.querySelector('.btn-menu');
        if (mobileMenuBtn) {
            mobileMenuBtn.addEventListener('click', function() {
                if (window.innerWidth <= 768) {
                    sidebar.classList.toggle('show');
                    if (overlay) {
                        overlay.classList.toggle('show');
                    }
                }
            });
        }
        
        // Close sidebar on window resize to desktop
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                sidebar.classList.remove('show');
                if (overlay) {
                    overlay.classList.remove('show');
                }
            }
        });
        
        // Highlight active page
        highlightActivePage();
        
        function highlightActivePage() {
            const currentPath = window.location.pathname + window.location.search;
            const currentFile = window.location.pathname.split('/').pop();
            
            // Find all menu links
            document.querySelectorAll('.sidebar-menu a[href]').forEach(link => {
                const linkHref = link.getAttribute('href');
                if (!linkHref || linkHref === '#') return;
                
                // Check for exact match
                if (linkHref === currentPath || linkHref.split('/').pop() === currentFile) {
                    // Add active class to link
                    link.classList.add('active');
                    
                    // Expand all parent dropdown menus
                    let parent = link.parentElement;
                    while (parent && !parent.classList.contains('sidebar-menu')) {
                        if (parent.tagName === 'LI' && (parent.classList.contains('dropdown') || parent.parentElement.classList.contains('dropdown-menu'))) {
                            parent.classList.add('open');
                        }
                        parent = parent.parentElement;
                    }
                }
            });
            console.log('✓ Active page highlighted');
        }
        
        console.log('🎉 Sidebar menu fully initialized!');
    }
})();
