<?php
session_start();
require_once 'auth_check.php';
require_once 'config.php';
require_once 'functions.php';

$license_id = isset($_GET['license_id']) ? $_GET['license_id'] : null;

if (!$license_id) {
    header('Location: licenses.php');
    exit;
}

$stmt = $pdo->prepare("SELECT l.*, c.client_name, c.company_name FROM licenses l JOIN clients c ON l.client_id = c.id WHERE l.id = ?");
$stmt->execute([$license_id]);
$license = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$license) {
    header('Location: licenses.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] == 'add') {
        $hardware_id = trim($_POST['hardware_id']);
        
        $check = $pdo->prepare("SELECT * FROM computers WHERE hardware_id = ?");
        $check->execute([$hardware_id]);
        if ($check->rowCount() > 0) {
            $error = "Hardware ID already exists in the system!";
        } else {
            $license_key = generateLicenseKey($hardware_id, $license['license_type'], $license['expiry_date']);
            
            $stmt = $pdo->prepare("INSERT INTO computers (license_id, computer_name, hardware_id, license_key, status, notes, created_by) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $license_id,
                $_POST['computer_name'],
                $hardware_id,
                $license_key,
                'pending',
                $_POST['notes'],
                $_SESSION['user_id']
            ]);
            $success = "Computer added successfully! License Key: " . $license_key;
        }
    } elseif ($_POST['action'] == 'edit') {
        $stmt = $pdo->prepare("UPDATE computers SET computer_name=?, notes=?, status=? WHERE id=?");
        $stmt->execute([
            $_POST['computer_name'],
            $_POST['notes'],
            $_POST['status'],
            $_POST['computer_id']
        ]);
        $success = "Computer updated successfully!";
    } elseif ($_POST['action'] == 'delete') {
        $stmt = $pdo->prepare("DELETE FROM computers WHERE id=?");
        $stmt->execute([$_POST['computer_id']]);
        $success = "Computer deleted successfully!";
    } elseif ($_POST['action'] == 'regenerate') {
        $computer = $pdo->prepare("SELECT * FROM computers WHERE id=?");
        $computer->execute([$_POST['computer_id']]);
        $comp = $computer->fetch(PDO::FETCH_ASSOC);
        
        $license_key = generateLicenseKey($comp['hardware_id'], $license['license_type'], $license['expiry_date']);
        
        $stmt = $pdo->prepare("UPDATE computers SET license_key=?, activated_at=NULL, status='pending' WHERE id=?");
        $stmt->execute([$license_key, $_POST['computer_id']]);
        $success = "License key regenerated! New Key: " . $license_key;
    }
}

$computers = $pdo->prepare("SELECT c.*, u.full_name as created_by_name FROM computers c LEFT JOIN users u ON c.created_by = u.id WHERE c.license_id = ? ORDER BY c.created_at DESC");
$computers->execute([$license_id]);
$computers = $computers->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Computer Management</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: Arial, sans-serif;
            background: #f5f5f5;
        }
        .navbar {
            background: #667eea;
            color: white;
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .navbar h1 {
            font-size: 24px;
        }
        .navbar a {
            color: white;
            text-decoration: none;
            margin-left: 20px;
            padding: 8px 15px;
            border-radius: 5px;
            background: rgba(255,255,255,0.2);
        }
        .navbar a:hover {
            background: rgba(255,255,255,0.3);
        }
        .container {
            max-width: 1400px;
            margin: 30px auto;
            padding: 0 20px;
        }
        .license-info {
            background: white;
            padding: 20px;
            border-radius: 5px;
            margin-bottom: 20px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .license-info h3 {
            margin-bottom: 10px;
        }
        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 15px;
        }
        .info-item {
            padding: 10px;
            background: #f9f9f9;
            border-radius: 5px;
        }
        .info-item strong {
            display: block;
            color: #667eea;
            margin-bottom: 5px;
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        .btn {
            padding: 10px 20px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }
        .btn:hover {
            background: #5568d3;
        }
        .btn-danger {
            background: #dc3545;
        }
        .btn-danger:hover {
            background: #c82333;
        }
        .btn-success {
            background: #28a745;
        }
        .btn-success:hover {
            background: #218838;
        }
        .btn-small {
            padding: 5px 10px;
            font-size: 14px;
            margin: 0 5px;
        }
        .success {
            background: #d4edda;
            color: #155724;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .error {
            background: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        table {
            width: 100%;
            background: white;
            border-radius: 5px;
            overflow: hidden;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        th {
            background: #667eea;
            color: white;
            font-size: 14px;
        }
        tr:hover {
            background: #f9f9f9;
        }
        .badge {
            padding: 5px 10px;
            border-radius: 3px;
            font-size: 12px;
            font-weight: bold;
        }
        .badge-active {
            background: #d4edda;
            color: #155724;
        }
        .badge-pending {
            background: #fff3cd;
            color: #856404;
        }
        .badge-inactive {
            background: #f8d7da;
            color: #721c24;
        }
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
        }
        .modal-content {
            background: white;
            margin: 50px auto;
            padding: 30px;
            border-radius: 10px;
            width: 90%;
            max-width: 600px;
        }
        .close {
            float: right;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
        }
        .form-group {
            margin-bottom: 15px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        input, select, textarea {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
        }
        textarea {
            resize: vertical;
            min-height: 80px;
        }
        .license-key-display {
            background: #f9f9f9;
            padding: 10px;
            border-radius: 5px;
            font-family: monospace;
            word-break: break-all;
            cursor: pointer;
        }
        .copy-btn {
            background: #28a745;
            color: white;
            border: none;
            padding: 5px 10px;
            border-radius: 3px;
            cursor: pointer;
            margin-left: 10px;
        }
    </style>
</head>
<body>
    <div class="navbar">
        <h1>License Management System</h1>
        <div>
            <a href="index.php">Dashboard</a>
            <a href="clients.php">Clients</a>
            <a href="licenses.php">Licenses</a>
            <a href="logout.php">Logout</a>
        </div>
    </div>

    <div class="container">
        <div class="license-info">
            <h3><?php echo htmlspecialchars($license['license_name']); ?></h3>
            <p><strong>Client:</strong> <?php echo htmlspecialchars($license['client_name'] . ' - ' . $license['company_name']); ?></p>
            
            <div class="info-grid">
                <div class="info-item">
                    <strong>License Type</strong>
                    <?php echo strtoupper($license['license_type']); ?>
                </div>
                <div class="info-item">
                    <strong>Start Date</strong>
                    <?php echo date('Y-m-d', strtotime($license['start_date'])); ?>
                </div>
                <div class="info-item">
                    <strong>Expiry Date</strong>
                    <?php echo $license['expiry_date'] ? date('Y-m-d', strtotime($license['expiry_date'])) : 'Never'; ?>
                </div>
                <div class="info-item">
                    <strong>Amount</strong>
                    <?php echo $license['currency'] . ' ' . number_format($license['amount'], 2); ?>
                </div>
                <div class="info-item">
                    <strong>Status</strong>
                    <?php echo strtoupper($license['status']); ?>
                </div>
                <div class="info-item">
                    <strong>Total Computers</strong>
                    <?php echo count($computers); ?>
                </div>
            </div>
            
            <div style="margin-top: 15px;">
                <a href="licenses.php?client_id=<?php echo $license['client_id']; ?>" class="btn btn-small">Back to Licenses</a>
            </div>
        </div>

        <div class="header">
            <h2>Computer Management</h2>
            <button class="btn" onclick="openAddModal()">Add New Computer</button>
        </div>

        <?php if (isset($success)): ?>
            <div class="success"><?php echo $success; ?></div>
        <?php endif; ?>
        
        <?php if (isset($error)): ?>
            <div class="error"><?php echo $error; ?></div>
        <?php endif; ?>

        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Computer Name</th>
                    <th>Hardware ID</th>
                    <th>License Key</th>
                    <th>Status</th>
                    <th>Created By</th>
                    <th>Activated</th>
                    <th>Last Check</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($computers) == 0): ?>
                <tr>
                    <td colspan="9" style="text-align: center; padding: 30px;">No computers added yet. Click "Add New Computer" to get started.</td>
                </tr>
                <?php else: ?>
                <?php foreach ($computers as $computer): ?>
                <tr>
                    <td><?php echo $computer['id']; ?></td>
                    <td><?php echo htmlspecialchars($computer['computer_name']); ?></td>
                    <td><code><?php echo htmlspecialchars(substr($computer['hardware_id'], 0, 20)) . '...'; ?></code></td>
                    <td>
                        <div class="license-key-display" onclick="copyToClipboard('<?php echo $computer['license_key']; ?>', this)">
                            <?php echo htmlspecialchars(substr($computer['license_key'], 0, 30)) . '...'; ?>
                            <button class="copy-btn">Copy</button>
                        </div>
                    </td>
                    <td><span class="badge badge-<?php echo $computer['status']; ?>"><?php echo strtoupper($computer['status']); ?></span></td>
                    <td><?php echo htmlspecialchars($computer['created_by_name']); ?></td>
                    <td><?php echo $computer['activated_at'] ? date('Y-m-d H:i', strtotime($computer['activated_at'])) : 'Not yet'; ?></td>
                    <td><?php echo $computer['last_checked'] ? date('Y-m-d H:i', strtotime($computer['last_checked'])) : 'Never'; ?></td>
                    <td>
                        <button class="btn btn-small" onclick='openEditModal(<?php echo json_encode($computer); ?>)'>Edit</button>
                        <button class="btn btn-small btn-success" onclick='showLicenseKey(<?php echo json_encode($computer); ?>)'>View Key</button>
                        <form method="POST" style="display:inline;" onsubmit="return confirm('Regenerate license key?');">
                            <input type="hidden" name="action" value="regenerate">
                            <input type="hidden" name="computer_id" value="<?php echo $computer['id']; ?>">
                            <button type="submit" class="btn btn-small">Regenerate</button>
                        </form>
                        <form method="POST" style="display:inline;" onsubmit="return confirm('Delete this computer?');">
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="computer_id" value="<?php echo $computer['id']; ?>">
                            <button type="submit" class="btn btn-small btn-danger">Delete</button>
                        </form>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <div id="computerModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeModal()">&times;</span>
            <h2 id="modalTitle">Add New Computer</h2>
            <form method="POST" id="computerForm">
                <input type="hidden" name="action" id="formAction" value="add">
                <input type="hidden" name="computer_id" id="computerId">
                
                <div class="form-group">
                    <label>Computer Name</label>
                    <input type="text" name="computer_name" id="computerName" placeholder="e.g., Reception PC, Accounting Laptop">
                </div>
                
                <div class="form-group" id="hardwareIdGroup">
                    <label>Hardware ID *</label>
                    <input type="text" name="hardware_id" id="hardwareId" placeholder="Paste Hardware ID from client" required>
                    <small style="color: #666;">Get this from the Python license server on the client's computer</small>
                </div>
                
                <div class="form-group" id="statusGroup" style="display:none;">
                    <label>Status</label>
                    <select name="status" id="status">
                        <option value="pending">Pending</option>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>Notes</label>
                    <textarea name="notes" id="notes" placeholder="Any additional information about this computer"></textarea>
                </div>
                
                <button type="submit" class="btn">Save Computer</button>
            </form>
        </div>
    </div>

    <div id="licenseKeyModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeLicenseKeyModal()">&times;</span>
            <h2>License Key Details</h2>
            <div style="margin: 20px 0;">
                <strong>Computer:</strong> <span id="keyComputerName"></span><br>
                <strong>Hardware ID:</strong><br>
                <div class="license-key-display" id="keyHardwareId" onclick="copyToClipboard(this.innerText, this)"></div>
                <br>
                <strong>License Key:</strong><br>
                <div class="license-key-display" id="keyLicenseKey" onclick="copyToClipboard(this.innerText, this)"></div>
            </div>
            <p style="color: #666; font-size: 14px;">Click on the key to copy to clipboard</p>
        </div>
    </div>

    <script>
        function openAddModal() {
            document.getElementById('modalTitle').innerText = 'Add New Computer';
            document.getElementById('formAction').value = 'add';
            document.getElementById('computerForm').reset();
            document.getElementById('hardwareIdGroup').style.display = 'block';
            document.getElementById('statusGroup').style.display = 'none';
            document.getElementById('hardwareId').setAttribute('required', 'required');
            document.getElementById('computerModal').style.display = 'block';
        }

        function openEditModal(computer) {
            document.getElementById('modalTitle').innerText = 'Edit Computer';
            document.getElementById('formAction').value = 'edit';
            document.getElementById('computerId').value = computer.id;
            document.getElementById('computerName').value = computer.computer_name;
            document.getElementById('notes').value = computer.notes;
            document.getElementById('status').value = computer.status;
            document.getElementById('hardwareIdGroup').style.display = 'none';
            document.getElementById('statusGroup').style.display = 'block';
            document.getElementById('hardwareId').removeAttribute('required');
            document.getElementById('computerModal').style.display = 'block';
        }

        function closeModal() {
            document.getElementById('computerModal').style.display = 'none';
        }

        function showLicenseKey(computer) {
            document.getElementById('keyComputerName').innerText = computer.computer_name || 'N/A';
            document.getElementById('keyHardwareId').innerText = computer.hardware_id;
            document.getElementById('keyLicenseKey').innerText = computer.license_key;
            document.getElementById('licenseKeyModal').style.display = 'block';
        }

        function closeLicenseKeyModal() {
            document.getElementById('licenseKeyModal').style.display = 'none';
        }

        function copyToClipboard(text, element) {
            navigator.clipboard.writeText(text).then(function() {
                const originalBg = element.style.background;
                element.style.background = '#d4edda';
                setTimeout(function() {
                    element.style.background = originalBg;
                }, 500);
            });
        }

        window.onclick = function(event) {
            if (event.target == document.getElementById('computerModal')) {
                closeModal();
            }
            if (event.target == document.getElementById('licenseKeyModal')) {
                closeLicenseKeyModal();
            }
        }
    </script>
</body>
</html>
