<?php
session_start();
require_once 'auth_check.php';
require_once 'config.php';

$stats = [];
$stats['total_clients'] = $pdo->query("SELECT COUNT(*) FROM clients")->fetchColumn();
$stats['total_licenses'] = $pdo->query("SELECT COUNT(*) FROM licenses")->fetchColumn();
$stats['active_licenses'] = $pdo->query("SELECT COUNT(*) FROM licenses WHERE status='active'")->fetchColumn();
$stats['expired_licenses'] = $pdo->query("SELECT COUNT(*) FROM licenses WHERE status='expired'")->fetchColumn();
$stats['total_computers'] = $pdo->query("SELECT COUNT(*) FROM computers")->fetchColumn();
$stats['active_computers'] = $pdo->query("SELECT COUNT(*) FROM computers WHERE status='active'")->fetchColumn();
$stats['total_revenue'] = $pdo->query("SELECT SUM(amount) FROM licenses")->fetchColumn() ?: 0;

$recent_licenses = $pdo->query("SELECT l.*, c.client_name, c.company_name FROM licenses l JOIN clients c ON l.client_id = c.id ORDER BY l.created_at DESC LIMIT 10")->fetchAll(PDO::FETCH_ASSOC);

$expiring_soon = $pdo->query("SELECT l.*, c.client_name FROM licenses l JOIN clients c ON l.client_id = c.id WHERE l.license_type='temporary' AND l.expiry_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY) ORDER BY l.expiry_date ASC LIMIT 5")->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - License Management</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: Arial, sans-serif;
            background: #f5f5f5;
        }
        .navbar {
            background: #667eea;
            color: white;
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .navbar h1 {
            font-size: 24px;
        }
        .navbar a {
            color: white;
            text-decoration: none;
            margin-left: 20px;
            padding: 8px 15px;
            border-radius: 5px;
            background: rgba(255,255,255,0.2);
        }
        .navbar a:hover {
            background: rgba(255,255,255,0.3);
        }
        .container {
            max-width: 1400px;
            margin: 30px auto;
            padding: 0 20px;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            text-align: center;
        }
        .stat-card h3 {
            font-size: 36px;
            color: #667eea;
            margin-bottom: 10px;
        }
        .stat-card p {
            color: #666;
            font-size: 14px;
        }
        .stat-card.revenue h3 {
            color: #28a745;
        }
        .section {
            background: white;
            padding: 25px;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        .section h2 {
            margin-bottom: 20px;
            color: #333;
        }
        table {
            width: 100%;
            border-collapse: collapse;
        }
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        th {
            background: #f8f9fa;
            font-weight: bold;
        }
        tr:hover {
            background: #f9f9f9;
        }
        .badge {
            padding: 5px 10px;
            border-radius: 3px;
            font-size: 12px;
            font-weight: bold;
        }
        .badge-active {
            background: #d4edda;
            color: #155724;
        }
        .badge-expired {
            background: #f8d7da;
            color: #721c24;
        }
        .badge-suspended {
            background: #fff3cd;
            color: #856404;
        }
        .badge-lifetime {
            background: #d1ecf1;
            color: #0c5460;
        }
        .badge-temporary {
            background: #e2e3e5;
            color: #383d41;
        }
        .badge-warning {
            background: #fff3cd;
            color: #856404;
        }
        .btn {
            padding: 8px 15px;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            font-size: 14px;
            display: inline-block;
        }
        .btn:hover {
            background: #5568d3;
        }
    </style>
</head>
<body>
    <div class="navbar">
        <h1>License Management System</h1>
        <div>
            <a href="index.php">Dashboard</a>
            <a href="clients.php">Clients</a>
            <a href="licenses.php">Licenses</a>
            <a href="logout.php">Logout</a>
        </div>
    </div>

    <div class="container">
        <h2 style="margin-bottom: 20px;">Dashboard Overview</h2>
        
        <div class="stats-grid">
            <div class="stat-card">
                <h3><?php echo $stats['total_clients']; ?></h3>
                <p>Total Clients</p>
            </div>
            <div class="stat-card">
                <h3><?php echo $stats['total_licenses']; ?></h3>
                <p>Total Licenses</p>
            </div>
            <div class="stat-card">
                <h3><?php echo $stats['active_licenses']; ?></h3>
                <p>Active Licenses</p>
            </div>
            <div class="stat-card">
                <h3><?php echo $stats['total_computers']; ?></h3>
                <p>Total Computers</p>
            </div>
            <div class="stat-card">
                <h3><?php echo $stats['active_computers']; ?></h3>
                <p>Active Computers</p>
            </div>
            <div class="stat-card revenue">
                <h3>$<?php echo number_format($stats['total_revenue'], 2); ?></h3>
                <p>Total Revenue</p>
            </div>
        </div>

        <?php if (count($expiring_soon) > 0): ?>
        <div class="section">
            <h2>Licenses Expiring Soon (Next 30 Days)</h2>
            <table>
                <thead>
                    <tr>
                        <th>License Name</th>
                        <th>Client</th>
                        <th>Expiry Date</th>
                        <th>Days Left</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($expiring_soon as $license): 
                        $days_left = ceil((strtotime($license['expiry_date']) - time()) / 86400);
                    ?>
                    <tr>
                        <td><?php echo htmlspecialchars($license['license_name']); ?></td>
                        <td><?php echo htmlspecialchars($license['client_name']); ?></td>
                        <td><?php echo date('Y-m-d', strtotime($license['expiry_date'])); ?></td>
                        <td><span class="badge badge-warning"><?php echo $days_left; ?> days</span></td>
                        <td>
                            <a href="licenses.php?client_id=<?php echo $license['client_id']; ?>" class="btn">View</a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>

        <div class="section">
            <h2>Recent Licenses</h2>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>License Name</th>
                        <th>Client</th>
                        <th>Type</th>
                        <th>Start Date</th>
                        <th>Expiry Date</th>
                        <th>Amount</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (count($recent_licenses) == 0): ?>
                    <tr>
                        <td colspan="9" style="text-align: center; padding: 30px;">
                            No licenses found. <a href="clients.php" class="btn">Add a client</a> to get started.
                        </td>
                    </tr>
                    <?php else: ?>
                    <?php foreach ($recent_licenses as $license): ?>
                    <tr>
                        <td><?php echo $license['id']; ?></td>
                        <td><?php echo htmlspecialchars($license['license_name']); ?></td>
                        <td><?php echo htmlspecialchars($license['client_name']); ?></td>
                        <td><span class="badge badge-<?php echo $license['license_type']; ?>"><?php echo strtoupper($license['license_type']); ?></span></td>
                        <td><?php echo date('Y-m-d', strtotime($license['start_date'])); ?></td>
                        <td><?php echo $license['expiry_date'] ? date('Y-m-d', strtotime($license['expiry_date'])) : 'Never'; ?></td>
                        <td><?php echo $license['currency'] . ' ' . number_format($license['amount'], 2); ?></td>
                        <td><span class="badge badge-<?php echo $license['status']; ?>"><?php echo strtoupper($license['status']); ?></span></td>
                        <td>
                            <a href="computers.php?license_id=<?php echo $license['id']; ?>" class="btn">View</a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</body>
</html>
