// JavaScript License Check - Include in every ERP page
// <script src="js_license_check.js"></script>

(function() {
    let licenseValid = false;
    let checkInProgress = false;
    
    async function checkLicense() {
        console.log('[LICENSE] Starting license check...');
        console.log('[LICENSE] checkInProgress:', checkInProgress);
        console.log('[LICENSE] document.body exists:', !!document.body);
        console.log('[LICENSE] document.readyState:', document.readyState);
        
        if (checkInProgress) {
            console.log('[LICENSE] Check already in progress, skipping');
            return false;
        }
        
        checkInProgress = true;
        
        try {
            console.log('[LICENSE] Fetching from server...');
            const response = await fetch('http://localhost:5050/check_license', {
                cache: 'no-cache'
            });
            
            console.log('[LICENSE] Response status:', response.status);
            
            if (!response.ok) {
                throw new Error('Server not reachable');
            }
            
            const data = await response.json();
            console.log('[LICENSE] Response data:', data);
            
            if (data.status === 'active' && data.details && data.details.valid) {
                console.log('[LICENSE] License is VALID');
                licenseValid = true;
                updateBanner(true, data.message);
                
                await fetch('save_license_status.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ valid: true, data: data })
                });
                
                checkInProgress = false;
                return true;
            } else {
                console.log('[LICENSE] License is INVALID');
                licenseValid = false;
                updateBanner(false, data.message);
                checkInProgress = false;
                console.log('[LICENSE] Calling blockPage()...');
                blockPage('License Error', data.message || 'Your license is invalid or expired. Please contact support.');
                return false;
            }
        } catch (error) {
            console.log('[LICENSE] ERROR:', error.message);
            licenseValid = false;
            updateBanner(false, 'License Server Not Running - Please start Elbaset License Manager');
            checkInProgress = false;
            console.log('[LICENSE] Calling blockPage() for server error...');
            blockPage('License Server Not Running', 'Please start the Elbaset License Manager application on your computer to access the system.');
            return false;
        }
    }
    
    function updateBanner(valid, message) {
        let banner = document.getElementById('license-banner');
        
        if (!banner) {
            banner = document.createElement('div');
            banner.id = 'license-banner';
            banner.style.cssText = 'position: fixed; top: 0; left: 0; right: 0; z-index: 9999; padding: 10px; text-align: center; font-weight: bold; color: white;';
            if (document.body) {
                document.body.insertBefore(banner, document.body.firstChild);
            }
        }
        
        if (valid) {
            banner.style.backgroundColor = '#4CAF50';
            banner.innerHTML = '✓ ' + message;
            setTimeout(() => {
                banner.style.display = 'none';
            }, 3000);
        } else {
            banner.style.backgroundColor = '#f44336';
            banner.innerHTML = '✗ ' + message;
        }
    }
    
    function blockPage(title, message) {
        console.log('[LICENSE] blockPage() called');
        console.log('[LICENSE] title:', title);
        console.log('[LICENSE] message:', message);
        console.log('[LICENSE] Existing overlay:', !!document.getElementById('license-block-overlay'));
        
        if (document.getElementById('license-block-overlay')) {
            console.log('[LICENSE] Overlay already exists, skipping');
            return;
        }
        
        console.log('[LICENSE] Creating style element...');
        const style = document.createElement('style');
        style.id = 'license-block-style';
        style.textContent = `
            html, body { 
                margin: 0 !important; 
                padding: 0 !important; 
                overflow: hidden !important; 
                height: 100% !important;
            }
            body > *:not(#license-block-overlay) {
                display: none !important;
            }
            #license-block-overlay {
                position: fixed !important;
                top: 0 !important;
                left: 0 !important;
                right: 0 !important;
                bottom: 0 !important;
                width: 100vw !important;
                height: 100vh !important;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
                z-index: 2147483647 !important;
                display: flex !important;
                justify-content: center !important;
                align-items: center !important;
                font-family: Arial, sans-serif !important;
                pointer-events: all !important;
            }
            #license-error-box {
                background: white !important;
                padding: 50px !important;
                border-radius: 10px !important;
                box-shadow: 0 10px 25px rgba(0,0,0,0.3) !important;
                max-width: 500px !important;
                text-align: center !important;
            }
            #license-error-box h1 {
                color: #333 !important;
                margin-bottom: 20px !important;
            }
            #license-error-box .retry-btn {
                padding: 12px 30px !important;
                background: #667eea !important;
                color: white !important;
                border: none !important;
                border-radius: 5px !important;
                cursor: pointer !important;
                font-size: 16px !important;
                font-weight: bold !important;
            }
            #license-error-box .retry-btn:hover {
                background: #5568d3 !important;
            }
        `;
        document.head.appendChild(style);
        console.log('[LICENSE] Style added to head');
        
        console.log('[LICENSE] Creating overlay element...');
        const overlay = document.createElement('div');
        overlay.id = 'license-block-overlay';
        overlay.innerHTML = `
            <div id="license-error-box">
                <div style="font-size: 72px; color: #dc3545; margin-bottom: 20px;">⚠</div>
                <h1>${title}</h1>
                <div style="background: #f8d7da; color: #721c24; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                    ${message}
                </div>
                <div style="text-align: left; background: #f9f9f9; padding: 20px; border-radius: 5px; margin-bottom: 20px;">
                    <h3 style="margin-bottom: 10px; color: #667eea;">To resolve this issue:</h3>
                    <ol style="margin-left: 20px;">
                        <li style="margin-bottom: 10px;"><strong>Start the License Server:</strong> Run the Elbaset License Manager application on your computer</li>
                        <li style="margin-bottom: 10px;"><strong>Check License Status:</strong> Make sure your license is active</li>
                        <li style="margin-bottom: 10px;"><strong>Contact Support:</strong> If your license has expired, contact support</li>
                    </ol>
                </div>
                <button class="retry-btn" onclick="location.reload()">Retry</button>
            </div>
        `;
        
        console.log('[LICENSE] document.body exists:', !!document.body);
        
        if (document.body) {
            console.log('[LICENSE] Adding overlay to body...');
            document.body.appendChild(overlay);
            console.log('[LICENSE] Overlay added to body');
        } else {
            console.log('[LICENSE] Body not ready, waiting for DOMContentLoaded...');
            window.addEventListener('DOMContentLoaded', () => {
                console.log('[LICENSE] DOMContentLoaded fired, adding overlay...');
                document.body.appendChild(overlay);
                console.log('[LICENSE] Overlay added to body');
            });
        }
    }
    
    window.addEventListener('DOMContentLoaded', checkLicense);
    if (document.readyState !== 'loading') {
        checkLicense();
    }
    
    setInterval(checkLicense, 300000);
})();
