<?php
session_start();
require_once 'auth_check.php';
require_once 'config.php';

$client_filter = isset($_GET['client_id']) ? $_GET['client_id'] : null;

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] == 'add') {
        $stmt = $pdo->prepare("INSERT INTO licenses (client_id, license_name, license_type, start_date, expiry_date, amount, currency, status, notes, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $expiry = ($_POST['license_type'] == 'lifetime') ? null : $_POST['expiry_date'];
        $stmt->execute([
            $_POST['client_id'],
            $_POST['license_name'],
            $_POST['license_type'],
            $_POST['start_date'],
            $expiry,
            $_POST['amount'],
            $_POST['currency'],
            $_POST['status'],
            $_POST['notes'],
            $_SESSION['user_id']
        ]);
        $success = "License created successfully!";
    } elseif ($_POST['action'] == 'edit') {
        $stmt = $pdo->prepare("UPDATE licenses SET client_id=?, license_name=?, license_type=?, start_date=?, expiry_date=?, amount=?, currency=?, status=?, notes=? WHERE id=?");
        $expiry = ($_POST['license_type'] == 'lifetime') ? null : $_POST['expiry_date'];
        $stmt->execute([
            $_POST['client_id'],
            $_POST['license_name'],
            $_POST['license_type'],
            $_POST['start_date'],
            $expiry,
            $_POST['amount'],
            $_POST['currency'],
            $_POST['status'],
            $_POST['notes'],
            $_POST['license_id']
        ]);
        $success = "License updated successfully!";
    } elseif ($_POST['action'] == 'delete') {
        $stmt = $pdo->prepare("DELETE FROM licenses WHERE id=?");
        $stmt->execute([$_POST['license_id']]);
        $success = "License deleted successfully!";
    }
}

if ($client_filter) {
    $stmt = $pdo->prepare("SELECT l.*, c.client_name, c.company_name FROM licenses l JOIN clients c ON l.client_id = c.id WHERE l.client_id = ? ORDER BY l.created_at DESC");
    $stmt->execute([$client_filter]);
    $licenses = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $client_info = $pdo->prepare("SELECT * FROM clients WHERE id = ?");
    $client_info->execute([$client_filter]);
    $client = $client_info->fetch(PDO::FETCH_ASSOC);
} else {
    $licenses = $pdo->query("SELECT l.*, c.client_name, c.company_name FROM licenses l JOIN clients c ON l.client_id = c.id ORDER BY l.created_at DESC")->fetchAll(PDO::FETCH_ASSOC);
}

$clients = $pdo->query("SELECT * FROM clients ORDER BY client_name")->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>License Management</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: Arial, sans-serif;
            background: #f5f5f5;
        }
        .navbar {
            background: #667eea;
            color: white;
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .navbar h1 {
            font-size: 24px;
        }
        .navbar a {
            color: white;
            text-decoration: none;
            margin-left: 20px;
            padding: 8px 15px;
            border-radius: 5px;
            background: rgba(255,255,255,0.2);
        }
        .navbar a:hover {
            background: rgba(255,255,255,0.3);
        }
        .container {
            max-width: 1400px;
            margin: 30px auto;
            padding: 0 20px;
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        .btn {
            padding: 10px 20px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }
        .btn:hover {
            background: #5568d3;
        }
        .btn-danger {
            background: #dc3545;
        }
        .btn-danger:hover {
            background: #c82333;
        }
        .btn-small {
            padding: 5px 10px;
            font-size: 14px;
            margin: 0 5px;
        }
        .success {
            background: #d4edda;
            color: #155724;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .client-info {
            background: white;
            padding: 20px;
            border-radius: 5px;
            margin-bottom: 20px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        table {
            width: 100%;
            background: white;
            border-radius: 5px;
            overflow: hidden;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        th {
            background: #667eea;
            color: white;
            font-size: 14px;
        }
        tr:hover {
            background: #f9f9f9;
        }
        .badge {
            padding: 5px 10px;
            border-radius: 3px;
            font-size: 12px;
            font-weight: bold;
        }
        .badge-active {
            background: #d4edda;
            color: #155724;
        }
        .badge-expired {
            background: #f8d7da;
            color: #721c24;
        }
        .badge-suspended {
            background: #fff3cd;
            color: #856404;
        }
        .badge-lifetime {
            background: #d1ecf1;
            color: #0c5460;
        }
        .badge-temporary {
            background: #e2e3e5;
            color: #383d41;
        }
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            overflow-y: auto;
        }
        .modal-content {
            background: white;
            margin: 50px auto;
            padding: 30px;
            border-radius: 10px;
            width: 90%;
            max-width: 700px;
        }
        .close {
            float: right;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
        }
        .form-group {
            margin-bottom: 15px;
        }
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        input, select, textarea {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
        }
        textarea {
            resize: vertical;
            min-height: 80px;
        }
        #expiryDateGroup {
            display: none;
        }
    </style>
</head>
<body>
    <div class="navbar">
        <h1>License Management System</h1>
        <div>
            <a href="index.php">Dashboard</a>
            <a href="clients.php">Clients</a>
            <a href="licenses.php">Licenses</a>
            <a href="logout.php">Logout</a>
        </div>
    </div>

    <div class="container">
        <?php if ($client_filter && isset($client)): ?>
        <div class="client-info">
            <h3><?php echo htmlspecialchars($client['client_name']); ?> - <?php echo htmlspecialchars($client['company_name']); ?></h3>
            <p>Email: <?php echo htmlspecialchars($client['email']); ?> | Phone: <?php echo htmlspecialchars($client['phone']); ?></p>
            <a href="licenses.php" class="btn btn-small">View All Licenses</a>
        </div>
        <?php endif; ?>

        <div class="header">
            <h2>License Management</h2>
            <button class="btn" onclick="openAddModal()">Create New License</button>
        </div>

        <?php if (isset($success)): ?>
            <div class="success"><?php echo $success; ?></div>
        <?php endif; ?>

        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>License Name</th>
                    <th>Client</th>
                    <th>Type</th>
                    <th>Start Date</th>
                    <th>Expiry Date</th>
                    <th>Amount</th>
                    <th>Status</th>
                    <th>Computers</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($licenses as $license): 
                    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM computers WHERE license_id = ?");
                    $stmt->execute([$license['id']]);
                    $computer_count = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
                ?>
                <tr>
                    <td><?php echo $license['id']; ?></td>
                    <td><?php echo htmlspecialchars($license['license_name']); ?></td>
                    <td><?php echo htmlspecialchars($license['client_name']); ?></td>
                    <td><span class="badge badge-<?php echo $license['license_type']; ?>"><?php echo strtoupper($license['license_type']); ?></span></td>
                    <td><?php echo date('Y-m-d', strtotime($license['start_date'])); ?></td>
                    <td><?php echo $license['expiry_date'] ? date('Y-m-d', strtotime($license['expiry_date'])) : 'Never'; ?></td>
                    <td><?php echo $license['currency'] . ' ' . number_format($license['amount'], 2); ?></td>
                    <td><span class="badge badge-<?php echo $license['status']; ?>"><?php echo strtoupper($license['status']); ?></span></td>
                    <td><?php echo $computer_count; ?></td>
                    <td>
                        <button class="btn btn-small" onclick='openEditModal(<?php echo json_encode($license); ?>)'>Edit</button>
                        <a href="computers.php?license_id=<?php echo $license['id']; ?>" class="btn btn-small">Computers</a>
                        <form method="POST" style="display:inline;" onsubmit="return confirm('Delete this license and all computers?');">
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="license_id" value="<?php echo $license['id']; ?>">
                            <button type="submit" class="btn btn-small btn-danger">Delete</button>
                        </form>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <div id="licenseModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeModal()">&times;</span>
            <h2 id="modalTitle">Create New License</h2>
            <form method="POST" id="licenseForm">
                <input type="hidden" name="action" id="formAction" value="add">
                <input type="hidden" name="license_id" id="licenseId">
                
                <div class="form-group">
                    <label>Client *</label>
                    <select name="client_id" id="clientId" required>
                        <option value="">Select Client</option>
                        <?php foreach ($clients as $c): ?>
                        <option value="<?php echo $c['id']; ?>"><?php echo htmlspecialchars($c['client_name'] . ' - ' . $c['company_name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>License Name *</label>
                    <input type="text" name="license_name" id="licenseName" placeholder="e.g., Main Office License" required>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>License Type *</label>
                        <select name="license_type" id="licenseType" onchange="toggleExpiryDate()" required>
                            <option value="temporary">Temporary</option>
                            <option value="lifetime">Lifetime</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Status *</label>
                        <select name="status" id="status" required>
                            <option value="active">Active</option>
                            <option value="expired">Expired</option>
                            <option value="suspended">Suspended</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Start Date *</label>
                        <input type="date" name="start_date" id="startDate" value="<?php echo date('Y-m-d'); ?>" required>
                    </div>
                    
                    <div class="form-group" id="expiryDateGroup">
                        <label>Expiry Date *</label>
                        <input type="date" name="expiry_date" id="expiryDate">
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Amount *</label>
                        <input type="number" name="amount" id="amount" step="0.01" value="0.00" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Currency *</label>
                        <select name="currency" id="currency" required>
                            <option value="USD">USD</option>
                            <option value="EUR">EUR</option>
                            <option value="GBP">GBP</option>
                            <option value="SAR">SAR</option>
                            <option value="AED">AED</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Notes</label>
                    <textarea name="notes" id="notes"></textarea>
                </div>
                
                <button type="submit" class="btn">Save License</button>
            </form>
        </div>
    </div>

    <script>
        function toggleExpiryDate() {
            const type = document.getElementById('licenseType').value;
            const expiryGroup = document.getElementById('expiryDateGroup');
            const expiryInput = document.getElementById('expiryDate');
            
            if (type === 'lifetime') {
                expiryGroup.style.display = 'none';
                expiryInput.removeAttribute('required');
            } else {
                expiryGroup.style.display = 'block';
                expiryInput.setAttribute('required', 'required');
            }
        }

        function openAddModal() {
            document.getElementById('modalTitle').innerText = 'Create New License';
            document.getElementById('formAction').value = 'add';
            document.getElementById('licenseForm').reset();
            document.getElementById('startDate').value = '<?php echo date('Y-m-d'); ?>';
            toggleExpiryDate();
            document.getElementById('licenseModal').style.display = 'block';
        }

        function openEditModal(license) {
            document.getElementById('modalTitle').innerText = 'Edit License';
            document.getElementById('formAction').value = 'edit';
            document.getElementById('licenseId').value = license.id;
            document.getElementById('clientId').value = license.client_id;
            document.getElementById('licenseName').value = license.license_name;
            document.getElementById('licenseType').value = license.license_type;
            document.getElementById('startDate').value = license.start_date;
            document.getElementById('expiryDate').value = license.expiry_date || '';
            document.getElementById('amount').value = license.amount;
            document.getElementById('currency').value = license.currency;
            document.getElementById('status').value = license.status;
            document.getElementById('notes').value = license.notes;
            toggleExpiryDate();
            document.getElementById('licenseModal').style.display = 'block';
        }

        function closeModal() {
            document.getElementById('licenseModal').style.display = 'none';
        }

        window.onclick = function(event) {
            if (event.target == document.getElementById('licenseModal')) {
                closeModal();
            }
        }

        toggleExpiryDate();
    </script>
</body>
</html>
